/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

// This file contains ssql data access implementation

///////////////////////////////////////////
/// Libraries
//////////////////////////////////////////
#pragma comment( lib, "odbccp32.lib" ) 

#include "hfiles.h"
#include "headers.h"
#include "mysql.h"
#include "mysqlmeta.h"
#include "ttypes.h"
#include "compare.h"
#include "odbcinst.h"   

#define CELL_STATE_INVALID	(-13)
#define MODE_UPDATE_SET		2
#define MODE_UPDATE_WHERE	1
#define MODE_INSERT			0
#define ROW_SET				1
#define ROW_WHERE			0
#define ROW_INSERT			0


DBTYPE GetOledbTypeFromName(LPOLESTR wszName, BOOL* pbIsMySqlBlob )
{
	TRACE2( "GetOledbTypeFromName, %S", wszName  );

	if( pbIsMySqlBlob )
		*pbIsMySqlBlob = FALSE;

	if (wszName == NULL)
		return DBTYPE_WSTR;

	if (_wcsicmp(wszName, L"INTEGER(1)") == 0 ||
		_wcsicmp(wszName, L"TINYINT")  == 0 ||
		_wcsicmp(wszName, L"DBTYPE_I1")  == 0)
	{
		return DBTYPE_I1;
	}

	if (_wcsicmp(wszName, L"INTEGER(2)") == 0 ||
		_wcsicmp(wszName, L"AUTOINC(2)") == 0 ||
		_wcsicmp(wszName, L"SMALLINT") == 0 ||
		_wcsicmp(wszName, L"SMALLINDENTITY") == 0 ||
		_wcsicmp(wszName, L"YEAR") == 0 ||
		_wcsicmp(wszName, L"DBTYPE_I2")  == 0)
	{
		return DBTYPE_I2;
	}

	if (_wcsicmp(wszName, L"INTEGER(4)") == 0 ||
		_wcsicmp(wszName, L"AUTOINC(4)") == 0 ||
		_wcsicmp(wszName, L"INTEGER")	 == 0 ||
		_wcsicmp(wszName, L"MEDIUMINT")	 == 0 ||
		_wcsicmp(wszName, L"INT")		 == 0 ||
		_wcsicmp(wszName, L"IDENTITY")	 == 0 ||
		_wcsicmp(wszName, L"DBTYPE_I4")	 == 0)
	{
		return DBTYPE_I4;
	}

	if (_wcsicmp(wszName, L"INTEGER(8)") == 0 ||
		_wcsicmp(wszName, L"BIGINT")	 == 0 ||
		_wcsicmp(wszName, L"DBTYPE_I8")  == 0)
	{
		return DBTYPE_I8;
	}

	if (_wcsicmp(wszName, L"FLOAT(4)")  == 0 ||
		_wcsicmp(wszName, L"BFLOAT(4)") == 0 ||
		_wcsicmp(wszName, L"REAL")		== 0 ||
		_wcsicmp(wszName, L"FLOAT")		== 0 ||
		_wcsicmp(wszName, L"FLOAT UNSIGNED")== 0 ||
		_wcsicmp(wszName, L"DBTYPE_R4") == 0)
	{
		return DBTYPE_R4;
	}

	if (_wcsicmp(wszName, L"FLOAT(8)")		== 0 ||
		_wcsicmp(wszName, L"BFLOAT(8)")		== 0 ||
		_wcsicmp(wszName, L"MONEY")			== 0 ||
		_wcsicmp(wszName, L"DECIMAL")		== 0 ||
		_wcsicmp(wszName, L"DOUBLE")		== 0 ||
		_wcsicmp(wszName, L"DOUBLE UNSIGNED")	== 0 ||
		_wcsicmp(wszName, L"DOUBLE PRECISION")	== 0 ||
		_wcsicmp(wszName, L"NUMERIC")		== 0 ||
		_wcsicmp(wszName, L"NUMERICSA")		== 0 ||
		_wcsicmp(wszName, L"NUMERICSTS")	== 0 ||
		_wcsicmp(wszName, L"DBTYPE_R8")		== 0)
	{
		return DBTYPE_R8;
	}


	if (_wcsicmp(wszName, L"CURRENCY")  == 0 ||
		_wcsicmp(wszName, L"DBTYPE_CY") == 0)
	{
		return DBTYPE_CY;
	}

	if (_wcsicmp(wszName, L"LOGICAL")	  == 0 ||
		_wcsicmp(wszName, L"BIT")		  == 0 ||
		_wcsicmp(wszName, L"DBTYPE_BOOL") == 0)
	{
		return DBTYPE_BOOL;
	}


	if (_wcsicmp(wszName, L"UNSIGNED(1)") == 0 ||
		_wcsicmp(wszName, L"DBTYPE_UI1")  == 0 ||
		_wcsicmp(wszName, L"UTINYINT")  == 0   ||
		_wcsicmp(wszName, L"TINYINT UNSIGNED")  == 0)
	{
		return DBTYPE_UI1;
	}

	if (_wcsicmp(wszName, L"UNSIGNED(2)") == 0 ||
		_wcsicmp(wszName, L"DBTYPE_UI2")  == 0 ||
		_wcsicmp(wszName, L"USMALLINT")  == 0  ||
		_wcsicmp(wszName, L"SMALLINT UNSIGNED")  == 0 )
	{
		return DBTYPE_UI2;
	}

	if (_wcsicmp(wszName, L"UNSIGNED(4)") == 0 ||
		_wcsicmp(wszName, L"DBTYPE_UI4")  == 0 ||
		_wcsicmp(wszName, L"UINTEGER")  == 0   ||
		_wcsicmp(wszName, L"INTEGER UNSIGNED")  == 0   ||
		_wcsicmp(wszName, L"MEDIUMINT UNSIGNED")  == 0   )
	{
		return DBTYPE_UI4;
	}

	if (_wcsicmp(wszName, L"UNSIGNED(8)") == 0 ||
		_wcsicmp(wszName, L"DBTYPE_UI8")  == 0 ||
		_wcsicmp(wszName, L"BIGINT UNSIGNED")  == 0 ||
		_wcsicmp(wszName, L"UBIGINT")  == 0)
	{
		return DBTYPE_UI8;
	}

	if (_wcsicmp(wszName, L"CHARACTER") == 0 ||
		_wcsicmp(wszName, L"LSTRING")	== 0 ||
		_wcsicmp(wszName, L"ZSTRING")	== 0 ||
		_wcsicmp(wszName, L"NOTE")		== 0 ||
		_wcsicmp(wszName, L"CHAR")		== 0 ||
		_wcsicmp(wszName, L"VARCHAR")		== 0 ||
		_wcsicmp(wszName, L"LONGVARCHAR")    == 0 ||
		_wcsicmp(wszName, L"NULL")    == 0 ||
		_wcsicmp(wszName, L"ENUM")		== 0 ||
		_wcsicmp(wszName, L"SET")		== 0 ||
		_wcsicmp(wszName, L"DBTYPE_STR") == 0)
	{
		return DBTYPE_STR;
	}

	if (_wcsicmp(wszName, L"TINYBLOB")		== 0 ||
		_wcsicmp(wszName, L"TINYTEXT")		== 0 ||
		_wcsicmp(wszName, L"BLOB")		== 0 ||
		_wcsicmp(wszName, L"TEXT")		== 0 ||
		_wcsicmp(wszName, L"MEDIUMBLOB")		== 0 ||
		_wcsicmp(wszName, L"MEDIUMTEXT")		== 0 ||
		_wcsicmp(wszName, L"LONGBLOB")		== 0 ||
		_wcsicmp(wszName, L"LONGTEXT")		== 0 )
	{
		if( pbIsMySqlBlob )
			*pbIsMySqlBlob = TRUE;
		return DBTYPE_STR;
	}

	if (_wcsicmp(wszName, L"LVAR")			== 0 ||
		_wcsicmp(wszName, L"LONG VARBINARY") == 0 ||
		_wcsicmp(wszName, L"LONGVARBINARY") == 0 ||
		_wcsicmp(wszName, L"BINARY") == 0 ||
		_wcsicmp(wszName, L"DBTYPE_BYTES")  == 0)
	{
		return DBTYPE_BYTES;
	}

	if (_wcsicmp(wszName, L"DATE")			== 0 ||
		_wcsicmp(wszName, L"DBTYPE_DBDATE") == 0)
	{
		return DBTYPE_DBDATE;
	}

	if (_wcsicmp(wszName, L"TIME")			== 0 ||
		_wcsicmp(wszName, L"DBTYPE_DBTIME") == 0)
	{
		return DBTYPE_DBTIME;
	}


	if (_wcsicmp(wszName, L"TIMESTAMP")			 == 0 ||
		_wcsicmp(wszName, L"DATETIME")			 == 0 ||
		_wcsicmp(wszName, L"DBTYPE_DBTIMESTAMP") == 0)
	{
		return DBTYPE_DBTIMESTAMP;
	}

	return DBTYPE_STR;
}



//Convert ANSI string to Unicode string
LPWSTR str2wstr(LPCTSTR str)
{
	int len = lstrlen(str);
	LPWSTR wstr = new WCHAR[len + 1];
	if (0 == MultiByteToWideChar( CP_ACP, MB_PRECOMPOSED, str, len, wstr, len))
	{
		delete [] wstr;
		return NULL;
	}

	wstr[len] = 0;
	return wstr;
}


//Skip relative
HRESULT CMySql::GetFetchedData(LONG lRow, DWORD* pdwBookmark)
{	
	if( lRow < 0 )
		return E_FAIL;
	if( m_hrLastMoveInfo != S_OK )
		return m_hrLastMoveInfo;

	m_dwCurrentRowInArray = lRow;

	if( Holder(this).Status(m_dwCurrentRowInArray) == SQL_ROW_NOROW )
		return DB_S_ENDOFROWSET; // behind end of rowset

	return GetBookmark( pdwBookmark );
}

// Invalidate buffers
HRESULT CMySql::SetMoveInfo( HRESULT hr )
{
	//m_dwValidRowsInArray = 0;
	m_hrLastMoveInfo = hr;
	return hr;
}

CMySql::Holder::Holder( CMySql* parent )
{
	m_parent = parent;
}

#define MYARRAY_EXT2 (sizeof(DWORD) + sizeof(SQLSMALLINT) + sizeof(SQLINTEGER) * m_parent->m_dwCols)

DWORD& CMySql::Holder::RowId( DWORD dwRow )
{
	return *(DWORD*)( m_parent->m_myArray + dwRow * sizeof(DWORD) );
}

SQLUSMALLINT& CMySql::Holder::Status( DWORD dwRow )
{
	return *(SQLUSMALLINT*)( m_parent->m_myArray + m_parent->m_dwTotalRowsInArray * sizeof(DWORD) + dwRow * sizeof(SQLUSMALLINT) );
}

SQLUSMALLINT* CMySql::Holder::StatusArray()
{
	return (SQLUSMALLINT*)( m_parent->m_myArray + m_parent->m_dwTotalRowsInArray * sizeof(DWORD) + m_parent->m_dwWasShifted * sizeof(SQLUSMALLINT) );
}

SQLINTEGER& CMySql::Holder::Hint( DWORD dwRow, DWORD dwCol )
{
	return *(SQLINTEGER*)( m_parent->m_myArray + m_parent->m_dwTotalRowsInArray * (sizeof(DWORD) + sizeof(SQLUSMALLINT) + sizeof(SQLINTEGER) * dwCol) + dwRow * sizeof(SQLINTEGER) );
}

SQLINTEGER* CMySql::Holder::HintArray( DWORD dwCol )
{
	return (SQLINTEGER*)( m_parent->m_myArray + m_parent->m_dwTotalRowsInArray * (sizeof(DWORD) + sizeof(SQLUSMALLINT) + sizeof(SQLINTEGER) * dwCol) + m_parent->m_dwWasShifted * sizeof(SQLINTEGER) );
}

BYTE* CMySql::Holder::RowColumn( DWORD dwRow, DWORD dwColumn )
{
	return m_parent->m_myArray + 
		m_parent->m_dwTotalRowsInArray * ( MYARRAY_EXT2 + m_parent->m_pColumnInfo[ dwColumn ].dwOffset ) + 
		dwRow * m_parent->m_pColumnInfo[ dwColumn ].uiLength;
}


HRESULT CMySql::Holder::Realloc( DWORD dwElems )
{
	DWORD dwWasTotalRowsInArray = m_parent->m_dwTotalRowsInArray;
	m_parent->m_dwTotalRowsInArray = 0;
	BYTE* new_buffer = new BYTE[ (dwElems+m_parent->m_dwValidRowsInArray)*(m_parent->m_dwRowSize + MYARRAY_EXT2) ];
	if( new_buffer == NULL )
		return E_OUTOFMEMORY;

	// Save new number of rows in the array
	m_parent->m_dwTotalRowsInArray = dwElems+m_parent->m_dwValidRowsInArray;
	if( m_parent->m_dwValidRowsInArray )
	{
		// Copy RowIds
		memcpy( new_buffer, m_parent->m_myArray, m_parent->m_dwValidRowsInArray * sizeof(DWORD) );
		// Copy Statuses
		memcpy( new_buffer + m_parent->m_dwTotalRowsInArray * sizeof(DWORD), m_parent->m_myArray + dwWasTotalRowsInArray * sizeof(DWORD), m_parent->m_dwValidRowsInArray * sizeof(SQLUSMALLINT) );
		// Copy hints & every column
		for( DWORD i = 0; i < m_parent->m_dwCols; i++ )
		{
			memcpy( new_buffer + m_parent->m_dwTotalRowsInArray * (sizeof(DWORD) + sizeof(SQLUSMALLINT) + sizeof(SQLINTEGER) * i), m_parent->m_myArray + dwWasTotalRowsInArray * (sizeof(DWORD) + sizeof(SQLUSMALLINT) + sizeof(SQLINTEGER) * i), m_parent->m_dwValidRowsInArray * sizeof(SQLINTEGER) );
			memcpy( new_buffer + m_parent->m_dwTotalRowsInArray * (MYARRAY_EXT2 + m_parent->m_pColumnInfo[i].dwOffset), m_parent->m_myArray + dwWasTotalRowsInArray * (MYARRAY_EXT2 + m_parent->m_pColumnInfo[i].dwOffset), m_parent->m_dwValidRowsInArray * m_parent->m_pColumnInfo[i].uiLength );
		}
	}
	
	// Switch pointers
	delete m_parent->m_myArray;
	m_parent->m_myArray = new_buffer;
	
	return S_OK;
}


//@cmember rebind buffers
HRESULT CMySql::ReBind(ULONG nRows, ULONG nShift)
{
	Holder i_am(this);

	// Invalidate rows in the buffer (but saved rows)
	m_dwValidRowsInArray = min( m_dwValidRowsInArray, nShift );

	// Do nothing if 0 rows to fetch
	if( !nRows )
		return S_FALSE;
	
	// Reallocate buffers if requested more rows
	if( m_dwTotalRowsInArray < nRows+nShift )
	{
		// Allocate, clone, apply, change number of rows, delete old
		HRESULT hr = Holder(this).Realloc(nRows);
		if( FAILED( hr ) )
			return hr;
		m_dwWasShifted = (DWORD)-1; // Impossible value to force rebind
	}

	// May be binded from 0th or 1st position
	if( m_dwWasShifted != nShift )
	{
		m_dwWasShifted = nShift;

		for( WORD i = 0; i < m_dwCols; i++ )
		{
			if( !SQL_SUCCEEDED( SQLBindCol( m_hstmt, i+1, m_pColumnInfo[i].iOdbcType, 
				Holder(this).RowColumn(nShift, i), m_pColumnInfo[i].uiLength, Holder(this).HintArray(i) ) ) )
					return E_FAIL;
		}
	}

	return S_OK;
}

//@cmember Move relative
HRESULT CMySql::MovePending(LONG nRows, BOOL bSaveCurrentPosition)
{
	Holder holder(this);
	UDWORD NumRows = 0, dummy;
	
	HRESULT hr = ReBind( abs( nRows ), bSaveCurrentPosition ? 1 : 0 );
	if( FAILED( hr ) )
		return m_hrLastMoveInfo = hr;
	if( hr == S_FALSE )
		return m_hrLastMoveInfo = S_OK; // no more to do

	if( !SQL_SUCCEEDED( SQLSetStmtOption( m_hstmt, SQL_ROWSET_SIZE, abs( nRows ) ) ) )
		return m_hrLastMoveInfo = E_FAIL;

	// Can fetch forwards only. dwFetchSince is number of first row (0-based) to fetch from. Cannot fetch before first row
	DWORD dwFetchSince = m_dwCurrentRow;
	if( nRows >= 0 )
		dwFetchSince ++;
	else
		dwFetchSince += nRows; // subtruction. Ok if (LONG)dwFetchSince become < 0 (it's handled by ODBC)

	// If nRows < 0 then -- nevertheless -- rows are fetched "ascending" to the array
	// So they will be enumerated in "revert" order. However they get correct RowId.
	// This mechanism differs with other recordsets of MySQL.OLEDB, where negative nRows cause "descending" enumeration.
	// I am not sure what way is correct. Fortunately, the difference appears only if nRows <= -2
	
	SQLRETURN ret = 
		SQLExtendedFetch( m_hstmt, SQL_FETCH_ABSOLUTE, 0, &dummy, NULL ) && // reset position; fortunately, is works quickly :(
		SQLExtendedFetch( m_hstmt, SQL_FETCH_RELATIVE, dwFetchSince, &NumRows, Holder(this).StatusArray() );
	if( SQL_SUCCEEDED( ret ) || ret == SQL_NO_DATA_FOUND )
		for( UDWORD i = m_dwValidRowsInArray; i < m_dwValidRowsInArray + NumRows; i++ )
		{
			m_dwCurrentRow += sgn( nRows );
			holder.RowId(i) = m_dwCurrentRow;
			CorrectData(i);
		}

	m_dwValidRowsInArray += NumRows; // ReBind sets it to 0 or 1

	if( ret == SQL_NO_DATA_FOUND || (SQL_SUCCEEDED( ret ) && !NumRows) )
	{
		// Correct a "feature" of MyODBC
		for( DWORD i = 0; i < abs( nRows ); i++ )
			holder.Status( i + bSaveCurrentPosition ? 1 : 0 ) = SQL_ROW_NOROW;

		m_hrLastMoveInfo = (bSaveCurrentPosition && m_hrLastMoveInfo==S_OK) ? S_OK : DB_S_ENDOFROWSET; // do not change saved S_OK!
		return DB_S_ENDOFROWSET; 
	}
	if( SQL_SUCCEEDED( ret ) )
	{
		m_hrLastMoveInfo = S_OK;
		return (NumRows < abs( nRows )) ? DB_S_ENDOFROWSET : S_OK;
	}
	return m_hrLastMoveInfo = E_FAIL;
}

//@cmember Move to first row
HRESULT CMySql::MoveFirst()
{
	Holder i_am(this);

	UDWORD NumRows = 0;
	
	HRESULT hr = ReBind( 1, 0 );
	if( FAILED( hr ) )
		return m_hrLastMoveInfo = hr;

	if( !SQL_SUCCEEDED( SQLSetStmtOption( m_hstmt, SQL_ROWSET_SIZE, 1 ) ) )
		return m_hrLastMoveInfo = E_FAIL;

	SQLRETURN ret =	SQL_SUCCEEDED( SQLExtendedFetch( m_hstmt, SQL_FETCH_FIRST, 0, &NumRows, Holder(this).StatusArray() ) );
	if( ret == SQL_NO_DATA_FOUND || (SQL_SUCCEEDED(ret) && !NumRows) )
		return m_hrLastMoveInfo = DB_S_ENDOFROWSET;

	if( SQL_SUCCEEDED(ret) )
	{
		Holder(this).RowId(0) = m_dwCurrentRow = 0;
		m_dwValidRowsInArray = 1;
		CorrectData(0);
		return m_hrLastMoveInfo = S_OK;
	}
	
	m_dwCurrentRow = (DWORD)-1;
	return m_hrLastMoveInfo = E_FAIL;		
}

//@cmember Move to last row
HRESULT CMySql::MoveLast()
{
	UDWORD NumRows = 0;
	
	HRESULT hr = ReBind( 1, 0 );
	if( FAILED( hr ) )
		return m_hrLastMoveInfo = hr;

	if( !SQL_SUCCEEDED( SQLSetStmtOption( m_hstmt, SQL_ROWSET_SIZE, 1 ) ) )
		return m_hrLastMoveInfo = E_FAIL;

	SQLRETURN ret =	SQL_SUCCEEDED( SQLExtendedFetch( m_hstmt, SQL_FETCH_LAST, 0, &NumRows, Holder(this).StatusArray() ) );
	if( ret == SQL_NO_DATA_FOUND || (SQL_SUCCEEDED(ret) && !NumRows) )
		return m_hrLastMoveInfo = DB_S_ENDOFROWSET;

	if( SQL_SUCCEEDED(ret) )
	{
		Holder(this).RowId(0) = m_dwCurrentRow = m_dwTotalRows - 1;
		m_dwValidRowsInArray = 1;
		CorrectData(0);
		return m_hrLastMoveInfo = S_OK;
	}
	
	m_dwCurrentRow = m_dwTotalRows;
	return m_hrLastMoveInfo = E_FAIL;		
}


//@cmember Move to specified bookmark
HRESULT CMySql::MoveBookmark(ULONG* pulBookmark)
{
	if( !pulBookmark )
		return E_INVALIDARG;
	
	UDWORD NumRows = 0;
	DWORD  pos = Bmk2Pos( *pulBookmark );
	
	HRESULT hr = ReBind( 1, 0 );
	if( FAILED( hr ) )
		return m_hrLastMoveInfo = hr;

	if( !SQL_SUCCEEDED( SQLSetStmtOption( m_hstmt, SQL_ROWSET_SIZE, 1 ) ) )
		return m_hrLastMoveInfo = E_FAIL;

	SQLRETURN ret =	SQL_SUCCEEDED( SQLExtendedFetch( m_hstmt, SQL_FETCH_ABSOLUTE, pos + 1, &NumRows, Holder(this).StatusArray() ) );
	if( ret == SQL_NO_DATA_FOUND || (SQL_SUCCEEDED(ret) && !NumRows) )
		return m_hrLastMoveInfo = DB_E_BADBOOKMARK;

	if( SQL_SUCCEEDED(ret) )
	{
		Holder(this).RowId(0) = m_dwCurrentRow = pos;
		m_dwValidRowsInArray = 1;
		m_dwCurrentRowInArray = 0;
		CorrectData(0);
		return m_hrLastMoveInfo = S_OK;
	}
	
	return m_hrLastMoveInfo = DB_E_BADBOOKMARK;		
}


//@cmember Find row from current position
HRESULT CMySql::Find(ULONG icol, PCOLUMNDATA pDst/*void* pvValue*/,	DBCOMPAREOP	CompareOp, DBTYPE dbType, bool bForward)
{
	Holder holder( this );

	if( m_dwCurrentRow >= m_dwValidRowsInArray ||
		holder.Status(m_dwCurrentRowInArray) == SQL_ROW_NOROW )
			return E_FAIL; // behind end of rowset

	DWORD dwUseRowInArray = m_dwCurrentRowInArray;
	void* pvValue = pDst->bData;
	bool bSuccess = false;

	
	// find for first row satisfied criteria
	while (true)
	{
		// if pvfield > pvvalue then 1
		void* pvField = holder.RowColumn(dwUseRowInArray, icol - 1);
		HRESULT hr;

		// comparison with NULL(s)?
		if( pDst->dwStatus == DBSTATUS_S_ISNULL || holder.Hint(dwUseRowInArray, icol - 1) == SQL_NULL_DATA )
		{
			int res;
			if( pDst->dwStatus == DBSTATUS_S_ISNULL && holder.Hint(dwUseRowInArray, icol - 1) == SQL_NULL_DATA )
				res = 0;
			else if( pDst->dwStatus == DBSTATUS_S_ISNULL )
				res = 1;
			else
				res = -1;
			hr = Compare(res, CompareOp, &bSuccess);
		}
		
		// compare
		else switch(dbType)
		{
		case DBTYPE_I8:
			hr = CompareNumerics((__int64 *)pvField, CompareOp, (PLONGLONG)pvValue, &bSuccess); 
			break;

		case DBTYPE_I4:
			hr = CompareNumerics((MT_LONG_PTR)pvField, CompareOp, (PLONG)pvValue, &bSuccess); 
			break;
		
		case DBTYPE_I2:
			hr = CompareNumerics((MT_SINT_PTR)pvField, CompareOp, (PSHORT)pvValue, &bSuccess); 
			break;
		
		case DBTYPE_I1:
			hr = CompareNumerics((signed char*)pvField, CompareOp, (PCHAR)pvValue, &bSuccess); 
			break;

		case DBTYPE_UI8:
			hr = CompareNumerics((unsigned __int64 *)pvField, CompareOp, (PULONGLONG)pvValue, &bSuccess); 
			break;

		case DBTYPE_UI4:
			hr = CompareNumerics((MT_ULONG_PTR)pvField, CompareOp, (PULONG)pvValue, &bSuccess); 
			break;
		
		case DBTYPE_UI2:
			hr = CompareNumerics((MT_WORD_PTR)pvField, CompareOp, (PUSHORT)pvValue, &bSuccess); 
			break;
		
		case DBTYPE_UI1:
			hr = CompareNumerics((unsigned char*)pvField, CompareOp, (PUCHAR)pvValue, &bSuccess); 
			break;
		
		case DBTYPE_R8:
			hr = CompareNumerics((double*)pvField, CompareOp, (double*)pvValue, &bSuccess); 
			break;

		case DBTYPE_R4:
			hr = CompareNumerics((float*)pvField, CompareOp, (float*)pvValue, &bSuccess); 
			break;

		/*case DBTYPE_CY:		
			hr = CompareNumerics((__int64*)pvField, CompareOp, &((LARGE_INTEGER*)pvValue)->QuadPart, &bSuccess); 
			break;*/

		case DBTYPE_BYTES:
		case DBTYPE_STR:
			{
				// Case sesitivity
				bool fCaseSensitive = (CompareOp & DBCOMPAREOPS_CASESENSITIVE)? true : false;
				bool fCaseInsensitive = (CompareOp & DBCOMPAREOPS_CASEINSENSITIVE)? true : false;
				CompareOp &= ~(DBCOMPAREOPS_CASESENSITIVE | DBCOMPAREOPS_CASEINSENSITIVE);

				// Those conditions are mutual exclusive so they must no be
				// simultanious
				if (fCaseSensitive && fCaseInsensitive)
					return DB_E_BADCOMPAREOP;
				// set default comparison mode
				else if( !fCaseSensitive && !fCaseInsensitive)
				{
					if( dbType == DBTYPE_BYTES )
						fCaseSensitive = true;
					else
						fCaseInsensitive = true;					
				}

				// perform comparision
				hr = CompareStrings((char*)pvField, CompareOp, pvValue, &bSuccess, fCaseSensitive); 
			}
			break;

		case DBTYPE_DBDATE:
		{
			DBDATE* oledbDate = (DBDATE*)pvValue;
			BT_DATE swstDate = { oledbDate->day, oledbDate->month, oledbDate->year };
			hr = CompareDates(pvField, CompareOp, &swstDate, &bSuccess); 
			break;
		}

		case DBTYPE_DBTIME:
		{
			DBTIME* oledbTime = (DBTIME*)pvValue;
			BT_TIME swstTime = { 0, oledbTime->second, oledbTime->minute, oledbTime->hour };
			hr = CompareTimes(pvField, CompareOp, &swstTime, &bSuccess); 
			break;
		}
		
		default:  // No more supported DBTYPEs
			hr = E_FAIL;
			bSuccess = false;
			break;
		}
		
		// Success or fatal failure?
		if (bSuccess)
			return S_OK;			
		if (hr != S_OK)
			return hr;
		
		// Next 1 row
		DWORD ulBookmark;
		hr = MovePending( bForward ? 1 : -1, FALSE );
		if( SUCCEEDED( hr ) )
			hr = GetFetchedData( 0, &ulBookmark );
		SetMoveInfo( hr );		
		if (hr != S_OK)
			return hr;

		dwUseRowInArray = 0;
	} 
}


//@cmember Return the number of rows in the table	
HRESULT CMySql::GetRowCnt(DWORD* pdwRows)
{
	if( !pdwRows )
		return E_INVALIDARG;

	*pdwRows = m_dwTotalRows;

	return S_OK;
}

//@cmember Get relative position (by percentage) of the row in  table
HRESULT CMySql::GetPercentage(ULONG* pulBookmark, double *pdfPPos)
{
	ULONG ulPos = Bmk2Pos(*pulBookmark);

	if (ulPos >= m_dwTotalRows)
		return E_INVALIDARG;

	*pdfPPos = double(ulPos) / double(m_dwTotalRows);
	
	return S_OK;
}

//@cmember Get bookmark for current row
HRESULT CMySql::GetBookmark(ULONG *pulBookmark)
{
	if( !pulBookmark )
		return E_INVALIDARG;

	// After insert, provider asks for a bookmark of a row, which is out of (static) result set
	if( m_enumLastChange == eINSERT_JUSTDONE )
	{
		m_enumLastChange = eINSERT;

		*pulBookmark = Pos2Bmk( m_dwTotalRows + m_dwRowsInserted );

		return S_OK;
	}

	if( m_dwCurrentRowInArray >= m_dwValidRowsInArray )
		return E_FAIL;
	
	*pulBookmark = Pos2Bmk( Holder(this).RowId(m_dwCurrentRowInArray) );

	return S_OK;
}

//@cmember Retrieve number of columns
HRESULT CMySql::GetColumnCnt(DWORD* pdwCount)
{
	if( !pdwCount )
		return E_INVALIDARG;

	*pdwCount = m_dwCols;

	return S_OK;
}

LPOLESTR str2wstr(LPCTSTR);

//@cmember Retrieve column information
HRESULT CMySql::GetColumnInfo(DWORD dwCol, DBCOLUMNINFO* pInfo)
{
	//Check arguments
	if (dwCol < 1 || dwCol > m_dwCols)
		return E_INVALIDARG;
	
	//Convert SQL type to OLEDB type
	WORD	wOledbType;			//OLEDB Type name
	DWORD	dwOledbFlags;		//OLEDB Type flags
	DWORD	dwOledbLength;		//OLEDB Type length	
	WORD	wOledbPrecision;	//OLEDB Type precision
	WORD	wOledbScale;		//OLEDB Type scale

	HRESULT hr;
	hr = CSwstMeta::SwstType2OledbTypeHelper
						(
						m_pColumnInfo[dwCol-1].wOleDbType, 
						m_pColumnInfo[dwCol-1].uiLength,
						m_pColumnInfo[dwCol-1].iDec,
						m_pColumnInfo[dwCol-1].iNullable,
						m_pColumnInfo[dwCol-1].iUpdatable,
						&wOledbType,
						&dwOledbFlags,
						&dwOledbLength,
						&wOledbPrecision,
						&wOledbScale
						);
	if (hr != S_OK)
		return hr;

	//Copy column info	
	pInfo->iOrdinal		= dwCol; 
    pInfo->pTypeInfo	= NULL; 
    pInfo->dwFlags		= dwOledbFlags; 
    pInfo->ulColumnSize = dwOledbLength; 
    pInfo->wType		= wOledbType;
    pInfo->bPrecision	= wOledbPrecision; 
    pInfo->bScale		= wOledbScale; 
    pInfo->columnid.eKind          = DBKIND_PROPID;
    pInfo->columnid.uGuid.guid     = GUID_NULL;
    pInfo->columnid.uName.ulPropid = pInfo->iOrdinal;

	// Copy name in special way
	pInfo->pwszName = str2wstr((const char*)m_pColumnInfo[dwCol-1].szName);

	return S_OK;
}

inline int read_hex( char* str )
{
	char szHex[] = { str[0], str[1], str[2], 0 };
	return strtol( szHex, NULL, 16 );
}

inline int read_oct( char* str )
{
	char szOct[] = { str[0], str[1], str[2], 0 };
	return strtol( szOct, NULL, 8 );
}

HRESULT CMySql::CorrectData(DWORD dwUseRowInArray)
{
	Holder holder(this);
	
	for (DWORD dwColumn = 0; dwColumn < m_dwCols; dwColumn++)
	{
		SQLINTEGER & iHint = holder.Hint(dwUseRowInArray, dwColumn);
		
		if( iHint == SQL_NULL_DATA )
			continue;
			
		if( iHint == SQL_NO_TOTAL )
			iHint = m_pColumnInfo[dwColumn].uiLength;

		// Correct some data types
		switch( m_pColumnInfo[dwColumn].wOleDbType )
		{
		case DBTYPE_I8:
		case DBTYPE_UI8:
			{
				char* p = (char*)holder.RowColumn(dwUseRowInArray, dwColumn);
				*(__int64*)p = _atoi64( p );
				iHint = sizeof(__int64);
			}
			break;
		case DBTYPE_STR:
			{
				char* pStart = (char*)holder.RowColumn(dwUseRowInArray, dwColumn);
				for( char* p = strchr( pStart, '\\' ); p && p < pStart + iHint + 1; p = strchr( p+1, '\\' ) )
					if( p[0] == '\\' && p[1] == '\\' ) // [\\] --> [\]
					{
						memcpy( p+1, p+2, (pStart+iHint+1)-p-2 );
						iHint--;
					}
					else if( p[0] == '\\' && p[1] == '0' ) // [\0] --> [eol]
					{
						p[0] = 0;
						memcpy( p+1, p+2, (pStart+iHint+1)-p-2 );
						iHint--;
					}
				break;
			}
		}
	}

	return S_OK;
}

//@cmember Fetch row data
HRESULT CMySql::GetRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	Holder holder(this);

	if( holder.Status(m_dwCurrentRowInArray) == SQL_ROW_NOROW )
		return E_FAIL; // behind end of rowset
	
	for (DWORD dwColumn = 0; dwColumn < m_dwCols; dwColumn++)
	{
		PCOLUMNDATA pColData = (PCOLUMNDATA) (pbProvRow + pulOffset[dwColumn+1]);
		SQLINTEGER iHint = holder.Hint(m_dwCurrentRowInArray, dwColumn);
		if( iHint == SQL_NULL_DATA )
		{
			pColData->dwStatus = DBSTATUS_S_ISNULL;
			pColData->dwLength = 0;
		}
		/*else if( iHint == SQL_NO_TOTAL ) // currently not used
		{
			pColData->dwStatus = DBSTATUS_S_TRUNCATED;
			pColData->dwLength = m_pColumnInfo[dwColumn].uiLength;
		}*/
		else
		{
			pColData->dwStatus = DBSTATUS_S_OK;
			pColData->dwLength = iHint;
		}

		if( iHint != SQL_NULL_DATA )
		{
			memcpy( pColData->bData, holder.RowColumn(m_dwCurrentRowInArray, dwColumn), 
				m_pColumnInfo[dwColumn].uiLength );
			
			if( pColData->dwLength < m_pColumnInfo[dwColumn].uiLength )
				if( m_pColumnInfo[dwColumn].uiLength <= 256 )
					memset( (BYTE*)pColData->bData + pColData->dwLength, 0, m_pColumnInfo[dwColumn].uiLength - pColData->dwLength );
				else // specially for BLOBs
					*((BYTE*)pColData->bData + pColData->dwLength) = 0;
		}
	}

    return S_OK;	
}

static DWORD fromAccessor( PACCESSOR pAccessor, DWORD i ) {	return pAccessor->rgBindings[i].iOrdinal; }

static DWORD fromOrdinal( PACCESSOR pAccessor, DWORD i ) {	return i + 1; }

	
// Apply an OLEDB buffer + accessor to internal buffer
HRESULT	CMySql::GetDataFromRow
	(
    DWORD		useMode, 
	ULONG*      ulOffset,   //@ parm IN | Array of offsets for the columns
    BYTE*       pbProvRow,  //@ parm IN | Data to compile a row.
	PACCESSOR   pAccessor,	//@ parm IN | Accessor with bindings.
	BOOL bOverwriteAutoinc,	//@ parm IN | Overwrite AUTOINC values
	ULONG		ulMaxBind	//@ parm IN | Maximum binding to apply
	)
{
	HRESULT hr = S_OK;
	Holder holder(this);

	// Nothing to do for null accessors
	if( pAccessor->bNullAccessor )
		return S_FALSE;
	
	DWORD		dwUseRowInAray;
	DWORD		dwLimit;
	DWORD		(*fOrdinal)(PACCESSOR, DWORD);

	switch( useMode )
	{
	case MODE_INSERT:
		dwUseRowInAray	= ROW_INSERT;
		dwLimit			= pAccessor->cBindings;
		fOrdinal		= fromAccessor;
		break;
	case MODE_UPDATE_SET:
		{
		dwUseRowInAray	= ROW_SET;
		dwLimit			= pAccessor->cBindings;
		fOrdinal		= fromAccessor;
		// Invallidate cells statuses
		for( WORD i = 0; i < m_dwCols; i++ )
			holder.Hint( dwUseRowInAray, i) = CELL_STATE_INVALID;
		break;
		}
	case MODE_UPDATE_WHERE:
		dwUseRowInAray	= ROW_WHERE;
		dwLimit			= m_dwCols;
		fOrdinal		= fromOrdinal;
		break;
	default:
		return S_FALSE;
	}
	

	// Overwrite ONLY bindings from accessor
	for (DWORD i = 0, ulBind = 0; i < dwLimit && ulBind < ulMaxBind && SUCCEEDED( hr ); i++)
	{
		DWORD iOrdinal = (*fOrdinal)( pAccessor, i );
		
		if( iOrdinal == 0 )
			continue;  // Exclude bookmarks

		// Not a self bookmark
		ulBind++;

		
		// Convert OLE DB buffer data to MyODBC buffer data
		// Correct some data types
		PCOLUMNDATA pColData = (PCOLUMNDATA) (pbProvRow + ulOffset[iOrdinal]);
		MYSQLCOLUMNINFO* pColInfo = m_pColumnInfo + (iOrdinal-1);
		BYTE* pPlace = holder.RowColumn( dwUseRowInAray, iOrdinal-1 );
		BYTE* pData = pColData->bData;
		SQLINTEGER& iSize = holder.Hint(dwUseRowInAray, iOrdinal-1);
		
		// Do not overwrite AUTOINCs
		if( !bOverwriteAutoinc && pColInfo->bIsAutoInc )
			continue;

		if( pColData->dwStatus == DBSTATUS_S_ISNULL )
			iSize = SQL_NULL_DATA;
		else
			switch( m_pColumnInfo[iOrdinal-1].wOleDbType )
			{
			case DBTYPE_I8:
				iSize = SQL_NTS;
				_i64toa( *(__int64*)pData, (char*)pPlace, 10 );
				break;
			case DBTYPE_UI8:
				iSize = SQL_NTS;
				_ui64toa( *(unsigned __int64*)pData, (char*)pPlace, 10 );
				break;
			case DBTYPE_STR:
				iSize = min(pColData->dwLength, pColInfo->uiLength - 1L);
				memcpy( pPlace, pData, iSize );
				break;
			default:
				iSize = min(pColData->dwLength, pColInfo->uiLength);
				memcpy( pPlace, pData, iSize );
				break;
			}
	}

	return hr;
}


//@cmember Update the current rows values
HRESULT CMySql::UpdateRow(
	ULONG*		pulBookmark,		//@parm IN | Bookmark of row to update	
    ULONG*      ulOffset,		//@parm IN | Array of offsets for the columns
    BYTE*       pbProvRow,		//@parm IN | Data to update row with.
	PACCESSOR	pAccessor,		//@parm IN | Accessor with bindings to update.
	BYTE*		pbProvRowOld	// old data
						 )
{
	if( !m_pUpdateFileInfo )
		return E_NOTIMPL; // cannot update anything (several tables used)
	
	Holder holder( this );

	// Ensure that there is space for two rows in internal buffer
	HRESULT hr = ReBind( 2, 0 );
	if( FAILED( hr ) )
		return hr;

	// We need to invalidate the array on exit from the function
	struct DoOnExit
	{
		DWORD& dwValidRowsInArray;
		DoOnExit( DWORD& vra ) : dwValidRowsInArray( vra ) {}
		~DoOnExit() { dwValidRowsInArray = 0; }
	} doOnExit (m_dwValidRowsInArray);

	hr = MoveBookmark( pulBookmark );
	if( FAILED(hr) )
		return hr;
	// some data may be changed! we use static cursors of mysql!
	hr = GetDataFromRow( MODE_UPDATE_WHERE, ulOffset, pbProvRowOld, pAccessor, TRUE );
	if( FAILED(hr) )
		return hr;

	BOOL bHasNulls = FALSE;
	for( DWORD w = 0; w < m_dwCols && !bHasNulls; w++ )
		if( (m_pColumnInfo[w].bInChangeQuery & eWHERE) && holder.HintIsNull(0, w ) )
			bHasNulls = TRUE;

	if( m_enumLastChange != eUPDATE || bHasNulls ) // A little optimization
	{
		m_enumLastChange = eNOTDONE;
		
		// Ensure that statement is set up.
		HRESULT ret = AllocChangeStmt();
		if( FAILED( ret ) )
			return ret;

		// Compose a query
		const char szUpdate[] = "update ", szSet[] = " set ", szEq[] = " = ?", szNextField[] = ", ";
		const char szWhere[] = " where ", szAnd[] = " and ", szIsNull[] = " is null";
		// a. calculate its length
		int iLen = 1 + MAXSTR(szUpdate) + strlen(m_pUpdateFileInfo->m_szName);
		for( DWORD w = 0, wCount = 0; w < m_dwCols; w++ )
			if( m_pColumnInfo[w].bInChangeQuery & eSET )
			{
				iLen += wCount ? MAXSTR(szNextField) : MAXSTR(szSet);
				iLen += strlen((char*)m_pColumnInfo[w].szName);
				iLen += MAXSTR(szEq);
				wCount++;
			}
		// Cannot be less than 1 field
		if( !wCount )
			return DB_E_ERRORSINCOMMAND;
		// where clause
		for( w = 0, wCount = 0; w < m_dwCols; w++ )
			if( m_pColumnInfo[w].bInChangeQuery & eWHERE )
			{
				iLen += wCount ? MAXSTR(szAnd) : MAXSTR(szWhere);
				iLen += strlen((char*)m_pColumnInfo[w].szName);
				iLen += holder.HintIsNull( 0, w ) ? MAXSTR(szIsNull) : MAXSTR(szEq);
				wCount++;
			}
		// Cannot be less than 1 field
		if( !wCount )
			return DB_E_ERRORSINCOMMAND;
			
		// b. allocate memory
		char* pszQuery = new char[ iLen ];
		if( !pszQuery )
			return E_OUTOFMEMORY;
		// c. Compose the query
		strcpy( pszQuery, szUpdate );
		strcat( pszQuery, m_pUpdateFileInfo->m_szName );
		for( w = 0, wCount = 0; w < m_dwCols; w++ )
			if( m_pColumnInfo[w].bInChangeQuery & eSET )
			{
				strcat( pszQuery, wCount ? szNextField : szSet );
				strcat( pszQuery, (char*)m_pColumnInfo[w].szName );
				strcat( pszQuery, szEq );
				wCount++;
			}
		// where clause
		for( w = 0, wCount = 0; w < m_dwCols; w++ )
			if( m_pColumnInfo[w].bInChangeQuery & eWHERE )
			{
				strcat( pszQuery, wCount ? szAnd : szWhere );
				strcat( pszQuery, (char*)m_pColumnInfo[w].szName );
				strcat( pszQuery, holder.HintIsNull( 0, w ) ? szIsNull : szEq );
				wCount++;
			}
	
		// Bind necessary parameters
		// a. prepare the statement
		ret = SQL_SUCCEEDED( SQLPrepare( m_hstmtChange, (BYTE*)pszQuery, SQL_NTS ) ) ? S_OK : DB_E_ERRORSINCOMMAND; 
		delete pszQuery; // no need anymore	
		if( FAILED(ret) )
			return ret;

		m_enumLastChange = bHasNulls ? eNOTDONE : eUPDATE;
	}


	// Apply an OLEDB buffer + accessor to internal buffer
	// bOverwriteAutoinc should be TRUE: shit cannot persist in the fields
	hr = GetDataFromRow( MODE_UPDATE_SET, ulOffset, pbProvRow, pAccessor, TRUE );
	if( FAILED( hr ) )
		return hr;

	// b. perform binding operation. NULLs are valid as parameters of UPDATE SET and invalid for UPDATE WHERE
	DWORD wParam = 1;
	for( w = 0; w < m_dwCols; w++ )
		if( m_pColumnInfo[w].bInChangeQuery & eSET )
		{
			int iFromRow = holder.Hint(1, w) == CELL_STATE_INVALID ? ROW_WHERE : ROW_SET;
			
			if( !SQL_SUCCEEDED( SQLBindParameter( m_hstmtChange, wParam, SQL_PARAM_INPUT, 
				m_pColumnInfo[w].iOdbcType, m_pColumnInfo[w].iOrigOdbcType,
				m_pColumnInfo[w].uiLength, m_pColumnInfo[w].iDec, 
				holder.RowColumn(iFromRow, w), 
				holder.Hint(iFromRow, w ), &holder.Hint(iFromRow, w) ) ) )
					return DB_E_ERRORSINCOMMAND;
			wParam++;
		}
	for( w = 0; w < m_dwCols; w++ )
		if( ( m_pColumnInfo[w].bInChangeQuery & eWHERE ) && !holder.HintIsNull(0, w) )
		{
			if( !SQL_SUCCEEDED( SQLBindParameter( m_hstmtChange, wParam, SQL_PARAM_INPUT, 
				m_pColumnInfo[w].iOdbcType, m_pColumnInfo[w].iOrigOdbcType,
				m_pColumnInfo[w].uiLength, m_pColumnInfo[w].iDec, 
				holder.RowColumn(ROW_WHERE, w), 
				holder.Hint(ROW_WHERE, w), &holder.Hint(ROW_WHERE, w) ) ) )
					return DB_E_ERRORSINCOMMAND;
			wParam++;
		}
	// c. execute the query
	if( !SQL_SUCCEEDED( SQLExecute( m_hstmtChange ) ) )
	{
		char szSqlState[ 10 ];
		if( SQL_SUCCEEDED( SQLError( m_henv, m_hdbc, m_hstmtChange, (BYTE*)szSqlState, NULL, NULL, 0, NULL ) ) &&
			!strcmp( szSqlState	, "23000" )	)
				return DB_E_INTEGRITYVIOLATION;

		return DB_E_ERRORSINCOMMAND;
	}

	// There may be 0 updated rows due to the shity optimization
/*
	// Check validity of the result
	SQLINTEGER iRowsAffected = 0;
	if( !SQL_SUCCEEDED( SQLRowCount( m_hstmtChange, &iRowsAffected ) ) || !iRowsAffected ) 
		return E_FAIL; ; // MySQL was not able to find the row
*/
	
	return S_OK;
}

//@cmember Insert new row
HRESULT CMySql::InsertRow(ULONG* ulOffset, BYTE* pbProvRow, PACCESSOR pAccessor)
{
	if( !m_pUpdateFileInfo )
		return E_NOTIMPL; // cannot update anything (several tables used)
	
	Holder holder( this );

	// Ensure that there is a row in internal buffer
	HRESULT hr = ReBind( 1, 0 );
	if( FAILED( hr ) )
		return hr;

	// We need to invalidate the array on exit from the function
	struct DoOnExit
	{
		DWORD& dwValidRowsInArray;
		DoOnExit( DWORD& vra ) : dwValidRowsInArray( vra ) {}
		~DoOnExit() { dwValidRowsInArray = 0; }
	} doOnExit (m_dwValidRowsInArray);
	
	if( m_enumLastChange != eINSERT && m_enumLastChange != eINSERT_JUSTDONE ) // A little optimization
	{
		m_enumLastChange = eNOTDONE;
		
		// Ensure that statement is set up.
		HRESULT ret = AllocChangeStmt();
		if( FAILED( ret ) )
			return ret;

		// Compose a query
		const char szInsertInto[] = "insert into ", szStartFields[] = " ( ", szNextField[] = ", ";
		const char szFirstValue[] = " ) values ( ?", szNextValue[] = ", ?", szTail[] = " )";
		// a. calculate its length
		int iLen = (MAXSTR(szTail) + 1) + MAXSTR(szInsertInto) + strlen(m_pUpdateFileInfo->m_szName);
		for( DWORD w = 0, wCount = 0; w < m_dwCols; w++ )
			if( m_pColumnInfo[w].bInChangeQuery & eSET )
			{
				iLen += wCount ? MAXSTR(szNextField) : MAXSTR(szStartFields);
				iLen += strlen((char*)m_pColumnInfo[w].szName);
				wCount++;
			}
		// Cannot be less than 1 field
		if( !wCount )
			return DB_E_ERRORSINCOMMAND;
		for( w = 0; w < wCount; w++ )
			iLen += w ? MAXSTR(szNextValue) : MAXSTR(szFirstValue);
			
		// b. allocate memory
		char* pszQuery = new char[ iLen ];
		if( !pszQuery )
			return E_OUTOFMEMORY;
		// c. Compose the query
		strcpy( pszQuery, szInsertInto );
		strcat( pszQuery, m_pUpdateFileInfo->m_szName );
		for( w = 0, wCount = 0; w < m_dwCols; w++ )
			if( m_pColumnInfo[w].bInChangeQuery & eSET )
			{
				strcat( pszQuery, wCount ? szNextField : szStartFields );
				strcat( pszQuery, (char*)m_pColumnInfo[w].szName);
				wCount++;
			}
		for( w = 0; w < wCount; w++ )
			strcat( pszQuery, w ? szNextValue : szFirstValue );
		strcat( pszQuery, szTail );
	
		// Bind necessary parameters
		// a. prepare the statement
		ret = SQL_SUCCEEDED( SQLPrepare( m_hstmtChange, (BYTE*)pszQuery, SQL_NTS ) ) ? S_OK : DB_E_ERRORSINCOMMAND; 
		delete pszQuery; // no need anymore	
		if( FAILED(ret) )
			return ret;

		m_enumLastChange = eINSERT;
	}

	// Apply an OLEDB buffer + accessor to internal buffer
	hr = GetDataFromRow( MODE_INSERT, ulOffset, pbProvRow, pAccessor, TRUE );
	if( FAILED( hr ) )
		return hr;

	// b. perform binding operation. NULLs are valid as parameters of INSERTs
	DWORD wParam = 1;
	for( DWORD w = 0; w < m_dwCols; w++ )
		if( m_pColumnInfo[w].bInChangeQuery & eSET )
		{
			if( !SQL_SUCCEEDED( SQLBindParameter( m_hstmtChange, wParam, SQL_PARAM_INPUT, 
				m_pColumnInfo[w].iOdbcType, m_pColumnInfo[w].iOrigOdbcType,
				m_pColumnInfo[w].uiLength, m_pColumnInfo[w].iDec, 
				holder.RowColumn(ROW_INSERT, w), 
				holder.Hint(ROW_INSERT, w ), &holder.Hint(ROW_INSERT, w) ) ) )
					return DB_E_ERRORSINCOMMAND;
			wParam++;
		}
	// c. execute the query
	if( !SQL_SUCCEEDED( SQLExecute( m_hstmtChange ) ) )
	{
		char szSqlState[ 10 ];
		if( SQL_SUCCEEDED( SQLError( m_henv, m_hdbc, m_hstmtChange, (BYTE*)szSqlState, NULL, NULL, 0, NULL ) ) &&
			!strcmp( szSqlState	, "23000" )	)
				return DB_E_INTEGRITYVIOLATION;

		return DB_E_ERRORSINCOMMAND;
	}

	// Check validity of the result
	SQLINTEGER iRowsAffected = 0;
	if( !SQL_SUCCEEDED( SQLRowCount( m_hstmtChange, &iRowsAffected ) ) || !iRowsAffected)
		return DB_E_INTEGRITYVIOLATION; // Invalid. Probably integrity violation

	// A trigger to get a bookmark for a row, which is out of result set
	m_enumLastChange = eINSERT_JUSTDONE;
	m_dwRowsInserted++;
	
	return S_OK;
}

//@cmember Remove row with the bmk
HRESULT CMySql::DeleteRow(ULONG* pulBookmark)
{
	if( !m_pUpdateFileInfo )
		return E_NOTIMPL; // cannot update anything (several tables used)
	
	Holder holder( this );

	// Save current row number & current row id
	// restore the position on exit
	struct DoOnExit
	{
		DWORD& dwRestoreCurrentRow;
		DWORD  dwRestoreCurrentRowVal;
		DWORD& dwValidRowsInArray;

		DoOnExit(DWORD& rcr, DWORD& vra ) : 
			dwRestoreCurrentRow(rcr),dwRestoreCurrentRowVal(rcr),dwValidRowsInArray(vra){}
		~DoOnExit() 
		{
			dwRestoreCurrentRow = dwRestoreCurrentRowVal;
			dwValidRowsInArray = 0;
		}
	} doOnExit( m_dwCurrentRow, m_dwValidRowsInArray );

	HRESULT hr = MoveBookmark( pulBookmark );
	if( FAILED( hr ) )
		return hr;

	BOOL bHasNulls = FALSE;
	for( DWORD w = 0; w < m_dwCols && !bHasNulls; w++ )
		if( (m_pColumnInfo[w].bInChangeQuery & eWHERE) && holder.HintIsNull(0, w ) )
			bHasNulls = TRUE;

	if( m_enumLastChange != eDELETE || bHasNulls ) // A little optimization
	{
		m_enumLastChange = eNOTDONE;
		
		// Ensure that statement is set up.
		HRESULT ret = AllocChangeStmt();
		if( FAILED( ret ) )
			return ret;

		// Compose a query
		const char szDeleteFrom[] = "delete from ";
		const char szWhere[] = " where ", szEq[] = " = ?", szIsNull[] = " is null", szAnd[] = " and ";
		// a. calculate its length
		int iLen = 1 + MAXSTR(szDeleteFrom) + strlen(m_pUpdateFileInfo->m_szName);
		for( DWORD w = 0, bNext = FALSE; w < m_dwCols; w++ )
			if( m_pColumnInfo[w].bInChangeQuery & eWHERE )
			{
				iLen += bNext ? MAXSTR(szAnd) : MAXSTR(szWhere);
				iLen += strlen((char*)m_pColumnInfo[w].szName);
				iLen += holder.HintIsNull( 0, w ) ? MAXSTR(szIsNull) : MAXSTR(szEq);
				bNext = TRUE;				
			}
		// b. allocate memory
		char* pszQuery = new char[ iLen ];
		if( !pszQuery )
			return E_OUTOFMEMORY;
		// c. Compose the query
		strcpy( pszQuery, szDeleteFrom );
		strcat( pszQuery, m_pUpdateFileInfo->m_szName );
		for( w = 0, bNext = FALSE; w < m_dwCols; w++ )
			if( m_pColumnInfo[w].bInChangeQuery & eWHERE )
			{
				strcat( pszQuery, bNext ? szAnd : szWhere );
				strcat( pszQuery, (char*)m_pColumnInfo[w].szName);
				strcat( pszQuery, holder.HintIsNull( 0, w ) ? szIsNull : szEq );
				bNext = TRUE;				
			}
	
		// Bind necessary parameters
		// a. prepare the statement
		ret = SQL_SUCCEEDED( SQLPrepare( m_hstmtChange, (BYTE*)pszQuery, SQL_NTS ) ) ? S_OK : DB_E_ERRORSINCOMMAND; 
		delete pszQuery; // no need anymore	
		if( FAILED(ret) )
			return ret;

		m_enumLastChange = bHasNulls ? eNOTDONE : eDELETE;
	}

	// b. perform binding operation
	DWORD wParam = 1;
	for( w = 0; w < m_dwCols; w++ )
		if( ( m_pColumnInfo[w].bInChangeQuery & eWHERE ) && !holder.HintIsNull(0, w) )
		{
			if( !SQL_SUCCEEDED( SQLBindParameter( m_hstmtChange, wParam, SQL_PARAM_INPUT, 
				m_pColumnInfo[w].iOdbcType, m_pColumnInfo[w].iOrigOdbcType,
				m_pColumnInfo[w].uiLength, m_pColumnInfo[w].iDec, 
				holder.RowColumn(0, w), 
				holder.Hint(0, w ), &holder.Hint(0, w) ) ) )
					return DB_E_ERRORSINCOMMAND;
			wParam++;
		}
	// cannot be less than 1 condition (else we delete *everything* in the table!)
	if( wParam == 1 )
		return DB_E_ERRORSINCOMMAND;
	// c. execute the query
	if( !SQL_SUCCEEDED( SQLExecute( m_hstmtChange ) ) )
		return DB_E_ERRORSINCOMMAND;

	// Check validity of the result
	SQLINTEGER iRowsAffected = 0;
	if( !SQL_SUCCEEDED( SQLRowCount( m_hstmtChange, &iRowsAffected ) ) || !iRowsAffected)
		return S_FALSE; // Invalid. Not done

	return S_OK;
}


CMySql::CMySql()
{
	CLEAR_CONSTRUCT(CMySql);
	
	m_dwCurrentRow = (DWORD)-1;
}


CMySql::~CMySql()
{
	if( m_hstmt != NULL )
		SQLFreeStmt( m_hstmt, SQL_DROP );
	if( m_hstmtChange != NULL )
		SQLFreeStmt( m_hstmtChange, SQL_DROP );
	if( m_hdbc != NULL )
	{
		SQLDisconnect( m_hdbc );
		SQLFreeConnect( m_hdbc );
	}
	if( m_henv != NULL )
		SQLFreeEnv( m_henv );

	delete m_myArray;
	delete m_pColumnInfo;
}


BOOL meta_modified( LPCSTR lpszSQL )
{
	char szBuf[ 101 ]; szBuf[ 100 ] = 0;
	for( const char* p = lpszSQL, *pEnd = lpszSQL + strlen(lpszSQL); p < pEnd; p += 90 )
	{
		strncpy( szBuf, p, 100 );
		_strupr( szBuf );
		if( strstr( szBuf, "CREATE" ) || 
			strstr( szBuf, "DROP" ) ||
			strstr( szBuf, "ALTER" ) )
				return TRUE;
	}
	return FALSE;
}


BOOL one_word( LPCSTR lpszSQL )
{
	// eat all starting spaces
	while( *lpszSQL == ' ' )
		lpszSQL++;
	// eat all non-spaces
	while( *lpszSQL && *lpszSQL != ' ' )
		lpszSQL++;
	// eat all ending spaces
	while( *lpszSQL == ' ' )
		lpszSQL++;
	// zero --> one word
	return !*lpszSQL;
}

void extract_table_name( char* pszDst, const char* pszSrc, int iMaxChars )
{
	*pszDst = 0;
	enum { eBEFORE, eFIRST, eINSIDE } state = eBEFORE;
	int iRestLen = iMaxChars;
	char szPart[ 256 ];
	for(;;)
	{ 
		if( sscanf( pszSrc, "%255s", szPart ) == EOF )
			break;
		if( state == eBEFORE && !stricmp( szPart, "FROM") )
			state = eFIRST;
		else if( state == eFIRST )
		{
			strcpy0( pszDst, szPart, iRestLen );
			iRestLen -= strlen( szPart );
			if( iRestLen < 0 ) iRestLen = 0;
			state = eINSIDE;
		}
		else if( state == eINSIDE && (!stricmp( szPart, "WHERE" ) || !stricmp( szPart, "ORDER" ) || !stricmp( szPart, "GROUP" ) ) )
			break;
		else if( state == eINSIDE )
		{
			strcpy0( pszDst + iMaxChars - iRestLen, " ",  iRestLen );
			iRestLen --;
			if( iRestLen < 0 ) iRestLen = 0;

			strcpy0( pszDst + iMaxChars - iRestLen, szPart, iRestLen );
			iRestLen -= strlen( szPart );
			if( iRestLen < 0 ) iRestLen = 0;
		}
		pszSrc += ( strstr( pszSrc, szPart ) - pszSrc ) + strlen( szPart );
	}
}


// Set up what fields to be used in SET and WHERE clauses
void CMySql::FindIndexFields()
{
	if( !m_pUpdateFileInfo )
		return; // cannot update anything (several tables used)
	
	const SWSTFILE* pFile = m_pUpdateFileInfo;
	MYSQLCOLUMNINFO* pColInfo = m_pColumnInfo;
	DWORD dwCols = m_dwCols;
	
	// Signs that the segment (field) is a part of the query
	MYSQLCOLUMNINFO** pSegments = new MYSQLCOLUMNINFO*[ pFile->m_wIndexes ];
	if( !pSegments ) 
		return;
	memset( pSegments, 0, pFile->m_wIndexes * sizeof(MYSQLCOLUMNINFO*));

	// Set up segments
	for( DWORD dw = 0; dw < dwCols; dw++ )
		for( WORD w = 0; w < pFile->m_wIndexes; w++ )
			if( !pSegments[w] &&
				!stricmp( (char*)pFile->m_pIndexes[w]->m_pSwstField->m_szName, (char*)pColInfo[dw].szName ) )
					pSegments[w] = pColInfo + dw;

	// Get a unique index, all fields of which are selected
	for( WORD w = 0, wIdx = -1, bValid = FALSE; w < pFile->m_wIndexes; w++ )
	{
		if( wIdx != pFile->m_pIndexes[w]->m_wNumber )
		{
			if( bValid )
				break;
			wIdx = pFile->m_pIndexes[w]->m_wNumber;
			bValid = TRUE;
		}
		if( bValid && ( !pSegments[w] || !pFile->m_pIndexes[w]->IsUnique() ) )
			bValid = FALSE;
	}

	// Mark all non-calculated fields as eSET. 
	// Mark the same fields (but BLOBs) as eWHERE if there is no unique index
	for( dw = 0; dw < dwCols; dw++ )
	{
		pColInfo[dw].bInChangeQuery = eNONE;
		
		for( w = 0; w < pFile->m_wFields; w++ )
			if( !stricmp( (char*)pFile->m_pFields[w]->m_szName, (char*)pColInfo[dw].szName ) )
			{
				if( pColInfo[dw].wOleDbType == DBTYPE_STR )
					pColInfo[dw].uiLength = max( pColInfo[dw].uiLength, pFile->m_pFields[w]->m_wSize + 1L );
				
				pColInfo[dw].bInChangeQuery = eSET;
				if( !bValid && !pColInfo[dw].bIsBlob )
					pColInfo[dw].bInChangeQuery |= eWHERE;
				break;
			}
	}

	// If there is a unique index, mark as eWHERE index fields only
	if( bValid ) 
	{
		for( w = 0; w < pFile->m_wIndexes; w++ )
			if( wIdx == pFile->m_pIndexes[w]->m_wNumber )
				pSegments[w]->bInChangeQuery |= eWHERE;
	}

	delete pSegments;
}

// Allocate a statement to change the recordset
HRESULT CMySql::AllocChangeStmt()
{
	if( m_hstmtChange != NULL )
		return FALSE; // nothing to do

	if( !m_pUpdateFileInfo )
		return E_NOTIMPL; // cannot update anything (several tables used)
	
	if( !SQL_SUCCEEDED( SQLAllocStmt( m_hdbc, &m_hstmtChange ) ) )
		return E_FAIL;
	if( !SQL_SUCCEEDED( SQLSetStmtOption( m_hstmtChange, SQL_BIND_TYPE, SQL_BIND_BY_COLUMN) ) )
		return E_FAIL;

	return S_OK;
}


//Initialize object with ssql query data
HRESULT CMySql::Init
	(
	CDataSource*	pDataSource,		//@parm IN	| DBSession
	LPCOLESTR		pwszCat,		// @parm IN | Use catalog( that is, DB). May be NULL or "" (Use current catalog in this case)
	LPCSTR			lpszSQL,		//@parm IN	| Text of SQL query
	DWORD*			pdwAffectedRows	//@parm OUT | Number of affected rows
	)
{
	HRESULT hr = S_OK;

	// 0) correct SQL syntax: tableName --> select * from tableName
	char _szSQL[ 256 ], __szSQL[ 256 ];
	LPCSTR szSQL = lpszSQL;
	if( one_word( lpszSQL ) )
	{
		const char szSelectFrom[] = "select * from ";
		strcpy( _szSQL, szSelectFrom );
		strcpy0( _szSQL + MAXSTR(szSelectFrom), lpszSQL, MAXSTR(_szSQL) - MAXSTR(szSelectFrom) );
		szSQL = _szSQL;
	}

	// We should obtain path of current catalogue
	// 1) Check pointers
	if( pDataSource->m_pUtilProp == NULL )
		return E_FAIL;

	// Save UtilProp pointer for any case
	CUtilProp* pUtilProp = pDataSource->m_pUtilProp;

	// 2) Get current catalog
	ULONG index;
	if( !pUtilProp->GetPropIndex(DBPROP_CURRENTCATALOG, &index ) )
		return E_FAIL;
	LPCOLESTR pwszCurCatalog = (pwszCat && *pwszCat) ? pwszCat : pUtilProp->m_rgproperties[ index ].pwstrVal;

	// 3) Get meta holder
	CSwstMetaHolder* pSwstMetaHolder;
	hr = pDataSource->GetSwstMeta(&pSwstMetaHolder);
    if (FAILED(hr))
        return hr;

	// 4) Get meta by catalog name. S_OK means that [out] pSwstMeta != NULL
	CSwstMeta* pSwstMeta;
	if( pSwstMetaHolder->At( pwszCurCatalog, 
							 NULL, 
							 &pSwstMeta ) != S_OK )
	{
		return E_FAIL;
	}

	// 5) Allocate all
	if( !SQL_SUCCEEDED( SQLAllocEnv( &m_henv ) ) )
		return E_FAIL;
	if( !SQL_SUCCEEDED( SQLAllocConnect( m_henv, &m_hdbc ) ) )
		return E_FAIL;
	if( !SQL_SUCCEEDED( SQLDriverConnect( m_hdbc, NULL, (BYTE*)pSwstMeta->m_szMySqlStr, SQL_NTS, NULL, 0, NULL, SQL_DRIVER_NOPROMPT ) ) )
		return E_FAIL;
	if( !SQL_SUCCEEDED( SQLAllocStmt( m_hdbc, &m_hstmt ) ) )
		return E_FAIL;
	if( !SQL_SUCCEEDED( SQLSetStmtOption( m_hstmt, SQL_CURSOR_TYPE, SQL_CURSOR_STATIC ) ) )
		return E_FAIL;
	
	// 6) Execute a query
	if( !SQL_SUCCEEDED( SQLExecDirect( m_hstmt, (BYTE*)szSQL, SQL_NTS ) ) )
		return DB_E_ERRORSINCOMMAND;

	// 7) Check what is done :) : SELECT returns one or more columns
	SQLSMALLINT  iCols;
	if( !SQL_SUCCEEDED( SQLNumResultCols( m_hstmt, &iCols ) ) )
		return E_FAIL;
	m_dwCols = iCols;

	// 8) Get number of rows affected
	SQLINTEGER	iRows;
	if( !SQL_SUCCEEDED( SQLRowCount( m_hstmt, &iRows ) ) )
		return E_FAIL;
	m_dwTotalRows = iRows;

	// 9) Non-SELECT? Return number of rows
	if( iCols == 0 )
	{
		*pdwAffectedRows = iRows;
		
		if( iRows == 0 && meta_modified( szSQL ) )
			pDataSource->ClearSwstMeta(); // force reloading of metadata
		
		return S_FALSE; // No rowset retrieved
	}

	// 10) SELECT
	*pdwAffectedRows = DB_COUNTUNAVAILABLE;	

	// Determine what table should be used for insert/update/delete operations (if available)
	extract_table_name( __szSQL, szSQL, MAXSTR(_szSQL) );
	if( one_word( __szSQL ) )
	{
		// copy trimmed table name
		char szUpdateTable[ 128 ];
		char* p = __szSQL, * pUpdate = szUpdateTable;
		while( *p == ' ' && *p ) 
			p++;
		while( *p != ' ' && *p && pUpdate - szUpdateTable < MAXSTR(szUpdateTable) ) 
			*pUpdate++ = *p++;
		*pUpdate = 0;

		m_pUpdateFileInfo = pSwstMeta->FindFile( szUpdateTable );
	}

	// Gather column information
	m_pColumnInfo = new MYSQLCOLUMNINFO[ m_dwCols ];
	if( m_pColumnInfo == NULL )
		return E_OUTOFMEMORY;

	SQLSMALLINT iLen;
	SQLUSMALLINT i = 0;
	DWORD dwOffset = 0;
	char szTypeName[ 50 ];
	WCHAR wszTypeName[ 50 ];
	SQLUINTEGER uiPrecision;
	SQLINTEGER  bUnsigned;
	while( SQL_SUCCEEDED( SQLDescribeCol( m_hstmt, i+1, m_pColumnInfo[i].szName, 
		sizeof(m_pColumnInfo[i].szName), &iLen, &m_pColumnInfo[i].iOrigOdbcType, 
		&uiPrecision, &m_pColumnInfo[i].iDec, &m_pColumnInfo[i].iNullable ) ) &&
		   SQL_SUCCEEDED( SQLColAttributes( m_hstmt, i+1, SQL_COLUMN_UPDATABLE, NULL, 0, NULL, 
		&m_pColumnInfo[i].iUpdatable ) ) &&
		   SQL_SUCCEEDED( SQLColAttributes( m_hstmt, i+1, SQL_COLUMN_LENGTH, NULL, 0, NULL, 
		(SQLINTEGER*)&m_pColumnInfo[i].uiLength ) ) &&
		   SQL_SUCCEEDED( SQLColAttributes( m_hstmt, i+1, SQL_COLUMN_TYPE_NAME, szTypeName, 
		sizeof(szTypeName), &iLen, NULL ) ) &&
		   SQL_SUCCEEDED( SQLColAttributes( m_hstmt, i+1, SQL_COLUMN_UNSIGNED, NULL, 0, NULL, 
		&bUnsigned ) ) &&
		   SQL_SUCCEEDED( SQLColAttributes( m_hstmt, i+1, SQL_COLUMN_AUTO_INCREMENT, NULL, 0, NULL, 
		&m_pColumnInfo[i].bIsAutoInc ) ) 
		)
	{
	    A2Wsz( szTypeName, wszTypeName );
		m_pColumnInfo[i].wOleDbType = GetOledbTypeFromName( wszTypeName, &m_pColumnInfo[i].bIsBlob );

		// Correct some ODBC types for good binding:
		switch( m_pColumnInfo[i].iOrigOdbcType )
		{
		case SQL_CHAR:
		case SQL_VARCHAR:
		case SQL_LONGVARCHAR:
		case SQL_LONGVARBINARY:
			m_pColumnInfo[i].iOdbcType = SQL_C_CHAR;
			m_pColumnInfo[i].uiLength++;
			break;
		case SQL_DECIMAL:
			m_pColumnInfo[i].iOdbcType = SQL_C_DOUBLE;
			m_pColumnInfo[i].uiLength = 8;
			break;
		case SQL_BIGINT:
			m_pColumnInfo[i].iOdbcType = SQL_C_CHAR;
			m_pColumnInfo[i].uiLength = 21;  // quite enough for string representation( including sign ) as well as for __int64
			break;
		case SQL_TINYINT:
		case SQL_SMALLINT:
		case SQL_INTEGER:
			m_pColumnInfo[i].iOdbcType = m_pColumnInfo[i].iOrigOdbcType;
			if( bUnsigned )
				m_pColumnInfo[i].iOdbcType += SQL_UNSIGNED_OFFSET;
			break;
		default:
			m_pColumnInfo[i].iOdbcType = m_pColumnInfo[i].iOrigOdbcType;
			break;
		}
		
		i++;
	}

	if( i != m_dwCols )
		return E_FAIL;

	FindIndexFields(); // fills m_pColumnInfo[i].bInChangeQuery, too

	for( i = 0; i < m_dwCols; i++ )
	{
		m_pColumnInfo[i].dwOffset = m_dwRowSize;
		m_dwRowSize += m_pColumnInfo[i].uiLength;	
	}

	return S_OK;
}
