/* Copyright (C) 2000 MySQL AB & MySQL Finland AB & TCX DataKonsult AB
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA */


/*
** PREPARE.C - This is the ODBC sample driver code for
** preparing SQL Commands and other functions prior to execution.
*/


#include "myodbc.h"
#ifndef _UNIX_
#include <dos.h>
#endif /* IS NOT UNIX */
#include <my_list.h>
#include <m_ctype.h>

//  Allocate a SQL statement

SQLRETURN SQL_API SQLAllocStmt(SQLHDBC hdbc,SQLHSTMT FAR *phstmt)
{
#ifndef _UNIX_
  HGLOBAL  hstmt;
#endif /* IS NOT UNIX */
  STMT FAR* stmt;
  DBC FAR *dbc=(DBC FAR*) hdbc;
  DBUG_ENTER("SQLAllocStmt");

#ifndef _UNIX_
  hstmt = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT, sizeof (STMT));
  if (!hstmt || (*phstmt = (SQLHSTMT)GlobalLock (hstmt)) == SQL_NULL_HSTMT)
  {
    GlobalFree (hstmt);	//	Free it if lock fails
    *phstmt = SQL_NULL_HSTMT;
    DBUG_RETURN(SQL_ERROR);
  }
#else
 *phstmt = (SQLHSTMT) my_malloc(sizeof (STMT), MYF(MY_ZEROFILL | MY_WME));
  if (*phstmt == SQL_NULL_HSTMT)
  {
    *phstmt = SQL_NULL_HSTMT;
    DBUG_RETURN(SQL_ERROR);
  }
#endif /* IS UNIX */
  stmt= (STMT FAR*) *phstmt;
  stmt->dbc= dbc;
  dbc->statements=list_add(dbc->statements,&stmt->list);
  stmt->list.data=stmt;
  stmt->stmt_options=dbc->stmt_options;
  init_dynamic_array(&stmt->params,sizeof(PARAM_BIND),32,64);
  DBUG_RETURN(SQL_SUCCESS);
}


/* We must use a wrapper here to allow other functions to call SQLFreeStmt */

SQLRETURN SQL_API SQLFreeStmt(SQLHSTMT hstmt,SQLUSMALLINT fOption)
{
  return my_SQLFreeStmt(hstmt,fOption);
}

SQLRETURN SQL_API my_SQLFreeStmt(SQLHSTMT hstmt,SQLUSMALLINT fOption)
{
  STMT FAR *stmt=(STMT FAR*) hstmt;
  uint i;
  DBUG_ENTER("SQLFreeStmt");
  DBUG_PRINT("enter",("stmt: %lx  option: %d",hstmt,fOption));

  if (fOption == SQL_UNBIND)
  {
    x_free(stmt->bind);
    stmt->bind=0;
    stmt->bound_columns=0;
    DBUG_RETURN(SQL_SUCCESS);
  }
  for (i=0 ; i < stmt->params.elements ; i++)
  {
    PARAM_BIND *param=dynamic_element(&stmt->params,i,PARAM_BIND*);
    if (param->alloced)
    {
      my_free(param->value,MYF(0));
      param->alloced=0;
    }
    if (fOption == SQL_RESET_PARAMS)
      param->used=0;
  }
  if (fOption == SQL_RESET_PARAMS)
  {
    DBUG_RETURN(SQL_SUCCESS);
  }

  mysql_free_result(stmt->result);
  x_free((gptr) stmt->fields);	     /* Don't write zero free's to debug log */
  x_free((gptr) stmt->array);
  x_free((gptr) stmt->result_array);
  x_free((gptr) stmt->odbc_types);
  stmt->result=0;
  stmt->result_lengths=0;
  stmt->fields=0;
  stmt->array=0;
  stmt->result_array=0;
  stmt->odbc_types=0;
  stmt->current_values=0;		/* For SQLGetData */
  stmt->fix_fields=0;
  stmt->current_row=stmt->rows_found_in_set=stmt->position_in_set=0;
  stmt->state=ST_UNKNOWN;
  /* Free data used by parameters */
  if (fOption != MYSQL_RESET_BUFFERS && fOption != SQL_CLOSE)
  {
    x_free((gptr) stmt->query);
    stmt->query=0;
    stmt->param_count=0;
  }
  if (fOption == SQL_DROP)
  {
    x_free((gptr) stmt->bind);
    delete_dynamic(&stmt->params);
    stmt->dbc->statements=list_delete(stmt->dbc->statements,&stmt->list);
#ifndef _UNIX_
    GlobalUnlock (GlobalHandle ((HGLOBAL) hstmt));
    GlobalFree (GlobalHandle((HGLOBAL) hstmt));
#endif /* IS NOT UNIX */

#ifdef _UNIX_
    my_free((char*) hstmt,MYF(0));
#endif /* IS UNIX */
  }
  DBUG_RETURN(SQL_SUCCESS);
}


//  Perform a Prepare on the SQL statement

SQLRETURN SQL_API SQLPrepare(SQLHSTMT hstmt,SQLCHAR FAR *szSqlStr,
			     SQLINTEGER cbSqlStr)
{
  return my_SQLPrepare(hstmt,szSqlStr,cbSqlStr);
}

SQLRETURN my_SQLPrepare(SQLHSTMT hstmt,SQLCHAR FAR *szSqlStr,
			SQLINTEGER cbSqlStr)
{
  STMT FAR *stmt=(STMT FAR*) hstmt;
  char in_string,*pos;
  uint param_count;
#ifdef  USE_MB
  char *end;
#endif
  CHARSET_INFO *charset_info=stmt->dbc->mysql.charset;
  DBUG_ENTER("my_SQLPrepare");

  if (stmt->query)
    my_free(stmt->query,MYF(0));
  /* SQLFreeStmt(hstmt,SQL_UNBIND); */ /* Not needed according to VB 5.0 */

  if (!(stmt->query=dupp_str((char*) szSqlStr, cbSqlStr)))
  {
    DBUG_RETURN(set_stmt_error(stmt,"S1001","Not enough memory",4001));
  }
  DBUG_PRINT("enter",("Query: %s",stmt->query));

  /* Count number of parameters and save position for each parameter */
  /* change also newline and return to space (for easy logging) */
  in_string=0;
  param_count=0;

#ifdef  USE_MB
  if (use_mb(charset_info))
    for (end=stmt->query; *end; end++) ;
#endif

  for (pos=stmt->query; *pos ; pos++)
  {
#ifdef  USE_MB
    if (use_mb(charset_info))
    {
      int l;
      if ((l = my_ismbchar(charset_info, pos , end )))
      {
	pos += l-1;
	continue;
      }
    }
#endif
    if (*pos == '\\' && pos[1])			/* Next char is escaped */
    {
      pos++;
      continue;
    }
    if (*pos == in_string)
    {
      if (pos[1] == in_string)			/* Two quotes is ok */
	pos++;
      else
	in_string=0;
      continue;
    }
    if (!in_string)
    {
      if (*pos == '\'' || *pos == '"' || *pos == '`')	/* Start of string */
      {
	in_string= *pos;
	continue;
      }
      if (*pos == '?')
      {
	PARAM_BIND *param;
	if (param_count >= stmt->params.elements)
	{
	  PARAM_BIND tmp_param;
	  bzero((gptr) &tmp_param,sizeof(tmp_param));
	  if (push_dynamic(&stmt->params,(gptr) &tmp_param))
	  {
	    DBUG_RETURN(set_stmt_error(stmt,"S1001","Not enough memory",4001));
	  }
	}
	param=dynamic_element(&stmt->params,param_count,PARAM_BIND*);
	param->pos_in_query=pos;
	param_count++;
      }
    }
  }
  stmt->param_count=param_count;
  stmt->query_end=pos;
  stmt->state=ST_PREPARED;
  DBUG_PRINT("exit",("Parameter count: %ld",stmt->param_count));
  DBUG_RETURN(SQL_SUCCESS);
}


//	Bind parameters on a statement handle

SQLRETURN SQL_API SQLBindParameter(SQLHSTMT hstmt,SQLUSMALLINT ipar,
				   SQLSMALLINT fParamType,
				   SQLSMALLINT fCType, SQLSMALLINT fSqlType,
				   SQLUINTEGER cbColDef,
				   SQLSMALLINT ibScale, SQLPOINTER  rgbValue,
				   SQLINTEGER cbValueMax,
				   SQLINTEGER FAR *pcbValue)
{
  STMT FAR *stmt=(STMT FAR*) hstmt;
  PARAM_BIND param;
  DBUG_ENTER("SQLBindParameter");
  DBUG_PRINT("enter",
	     ("ipar: %d  Ctype: %d  SQLtype: %d  rgbValue: %lx  ValueMax: %ld  Valueptr: %lx  Value: %ld",
	      ipar,fCType,fSqlType,rgbValue, cbValueMax,
	      pcbValue, pcbValue ? *pcbValue : 0L));

  if (ipar-- < 1)
  {
    DBUG_RETURN(set_stmt_error(stmt,"S1093","Invalid parameter number",0));
  }
  if (stmt->params.elements > ipar)
  {				/* Change old bind parameter */
    PARAM_BIND *old=dynamic_element(&stmt->params,ipar,PARAM_BIND*);
    if (old->alloced)
    {
      my_free(old->value,MYF(0));
      old->alloced=0;
    }
    memcpy((gptr) &param,(gptr) old,sizeof(param));
  }
  else
    bzero(&param, sizeof(param));
  /* Simply record the values. These are used later (SQLExecute) */
  param.used= 1;
  param.SqlType = fSqlType;
  param.CType = (fCType == SQL_C_DEFAULT ? default_c_type(fSqlType) : fCType);
  param.buffer = rgbValue;
  param.ValueMax = cbValueMax;
  param.actual_len= pcbValue;
  if (set_dynamic(&stmt->params,(gptr) &param,ipar))
  {
    DBUG_RETURN(set_stmt_error(stmt,"S1001","Not enough memory",4001));
  }
  DBUG_RETURN(SQL_SUCCESS);
}


/*
** Returns the description of a parameter marker.
** As we can't detect this easily, but MySQL can use strings anywhere
** we simple say that all parameters can be strings.
*/

SQLRETURN SQL_API SQLDescribeParam(SQLHSTMT hstmt, SQLUSMALLINT ipar,
				   SQLSMALLINT FAR *pfSqlType,
				   SQLUINTEGER FAR *pcbColDef,
				   SQLSMALLINT FAR *pibScale,
				   SQLSMALLINT FAR *pfNullable)
{
  STMT FAR *stmt=(STMT FAR*) hstmt;
  DBUG_ENTER("SQLDescribeParam");
  if (pfSqlType)
    *pfSqlType=SQL_VARCHAR;
  if (pcbColDef)
    *pcbColDef= (stmt->dbc->flag & FLAG_BIG_PACKETS ? 24*1024*1024L : 255);
  if (pfNullable)
    *pfNullable=SQL_NULLABLE_UNKNOWN;
  DBUG_RETURN(SQL_SUCCESS);
}


//  Sets multiple values (arrays) for the set of parameter markers.

SQLRETURN SQL_API SQLParamOptions(SQLHSTMT hstmt, SQLUINTEGER crow,
				  SQLUINTEGER FAR *pirow)
{
  DBUG_ENTER("SQLParamOptions");
  DBUG_PRINT("info",("Command ignored"));
  DBUG_RETURN(SQL_SUCCESS);
}


//  Returns the number of parameter markers.

SQLRETURN SQL_API SQLNumParams(SQLHSTMT hstmt, SQLSMALLINT FAR *pcpar)
{
  STMT FAR *stmt=(STMT FAR*) hstmt;
  DBUG_ENTER("SQLNumParams");
  if (pcpar)
    *pcpar=stmt->param_count;
  DBUG_RETURN(SQL_SUCCESS);
}


//  Sets options that control the behavior of cursors.

SQLRETURN SQL_API SQLSetScrollOptions(SQLHSTMT hstmt,
				      SQLUSMALLINT fConcurrency,
				      SQLINTEGER crowKeyset,
				      SQLUSMALLINT crowRowset)
{
  STMT FAR *stmt=(STMT FAR*) hstmt;
  DBUG_ENTER("SQLSetScrollOptions");
  stmt->stmt_options.rows_in_set=crowRowset;
  DBUG_RETURN(SQL_SUCCESS);
}


//  Set the cursor name on a statement handle

SQLRETURN SQL_API SQLSetCursorName(SQLHSTMT  hstmt, SQLCHAR FAR *szCursor,
				 SQLSMALLINT cbCursor)
{
  DBUG_ENTER("SQLSetCursorName");
  DBUG_PRINT("info",("Command ignored as MyODBC doesn't support cursors"));
  DBUG_RETURN(SQL_SUCCESS);
}


//  Return the cursor name for a statement handle

SQLRETURN SQL_API SQLGetCursorName(SQLHSTMT hstmt, SQLCHAR FAR *szCursor,
				   SQLSMALLINT cbCursorMax,
				   SQLSMALLINT FAR *pcbCursor)
{
  DBUG_ENTER("SQLGetCursorName");
  copy_result(((STMT FAR*) hstmt)->dbc,szCursor,cbCursorMax,pcbCursor,
	      "NO_NAME");
  DBUG_RETURN(SQL_SUCCESS);
}
