/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

#ifndef _cache_val_h_
#define _cache_val_h_

#include <vector>
#include <list>
#include <assert.h>

template< class Val, class Param >
class CacheVal
{
public:
    /** 
     * constructor 
     *
     * @param _reliability_interval [in] reliability time for cached value, default - 1min
     * @param _entries [in] maximum number of elements simultaneously in cache
     */
    CacheVal( int _entries = 10, __int64 _reliability_interval = 600000000 ) 
        : reliability_interval(_reliability_interval), entries(_entries) {}

    /** main: updates val only if params differs or too much time passed since it set up */
    Val& update( Param &new_param )
    {
        //determine current time
        SYSTEMTIME SystemTime;
        GetSystemTime( &SystemTime );

        union{
            FILETIME FileTime;
            __int64 now;
        } u;
        SystemTimeToFileTime( &SystemTime, &u.FileTime );

        //cache already contains required element - no need in slow function
        LRUType::iterator found = LRU.end(); //while not

        //go through all values stored in cache
        LRUType::iterator l;
        for( l = LRU.begin(); l != LRU.end(); ) {
            if( u.now > (*l)->was_set + reliability_interval ) { 
                //value is too old - remove it

                //from entries
                (*l)->value.release();
                (*l)->was_set = 0; //mark free

                //from LRU
                LRUType::iterator l1 = l++;
                LRU.erase( l1 );

                continue;
            }

            if( (*l)->key == new_param ) {
                found = l;
            }

            ++l;
        }

        //if really found
        if( found != LRU.end() ) {
            //reorganize LRU - found element should start the list
            LRU.insert( LRU.begin(), *found );
            LRU.erase( found );

            new_param = (*LRU.begin())->key;
            return (*LRU.begin())->value;
        }

//        tlog.init(0,0,-1 & ~MSG_THREAD_ID ); 
//        tlog << setlock << settype(MSG_FATAL_ERROR) << "miss" << release;

        //find where we can place new entry
        EntriesType::iterator i;
        if( LRU.size() < entries.size() ) {
            //cache is not filled yet up to maximum capacity

            //find first unoccupied entry
            for( i = entries.begin(); i < entries.end(); ++i ) {
                if( i->was_set == 0 ) break;
            }
            assert( i != entries.end() );
        } else {
            //cache already full -> we should remove one entry - that was LRU
            LRUType::iterator l1 = LRU.end(); --l1;
            i = *l1;
            LRU.erase( l1 );
            i->value.release();
        }

        //obtain and store new value
        i->value.update( new_param );
        i->key = new_param;
        i->was_set = u.now;

        //reorganize LRU - new element should start the list
        LRU.insert( LRU.begin(), i );

        return i->value;
    }

    /** release all cached values: used on program exit */
    void release_all() {
        EntriesType::iterator i;
        for( i = entries.begin(); i < entries.end(); ++i ) {
            if( i->was_set != 0 ) {
                i->value.release();
                i->was_set = 0;
            }
        }
        LRU.clear();
    }

private:
    /** element of cache */
    struct Entry {
        Entry() : was_set(0) {}

        Param key;
        Val value;

        /** when current value was cached */
        __int64 was_set; //if ==0 then this is empty entry
    };

    /** cached values */
    typedef std::vector<Entry> EntriesType;
    EntriesType entries;

    /** least-recently-use - the last element, in list indexes of entries are stored */
    typedef std::list<EntriesType::iterator> LRUType;
    LRUType LRU;

    /** the number of 100-nanosecond intervals during which we can use cached value withou update */
    __int64 reliability_interval;
};

#endif //_cache_val_h_
