/*
   Copyright (c) 2004, 2014, Oracle and/or its affiliates. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2 of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
*/

#ifndef NDB_VERSION_H
#define NDB_VERSION_H

#include <ndb_types.h>
#include <mysql_version.h>

/*
  Creates a composite version number from major, minor and build
  ex: NDB_MAKE_VERSION(5,1,47) => 0x00050147
*/
#define NDB_MAKE_VERSION(A,B,C) (((A) << 16) | ((B) << 8)  | ((C) << 0))

/*
  Creates a stringified version from major, minor and build
  ex: NDB_MAKE_STRING_VERSION(7,0,22) => "7.0.22"
*/
#define NDB_MAKE_QUOTED_VERSION(A,B,C) #A "." #B "." #C
#define NDB_MAKE_STRING_VERSION(A,B,C) NDB_MAKE_QUOTED_VERSION(A,B,C)

/* NDB version numbers and status  */
#define NDB_VERSION_MAJOR 7
#define NDB_VERSION_MINOR 2
#define NDB_VERSION_BUILD 34
#define NDB_VERSION_STATUS ""

/* Composite version number for NDB */
#define NDB_VERSION_D \
  NDB_MAKE_VERSION(NDB_VERSION_MAJOR, NDB_VERSION_MINOR, NDB_VERSION_BUILD)

/* Version string for NDB, ex: "ndb-7.0.22" */
#define NDB_NDB_VERSION_STRING \
  "ndb-" NDB_MAKE_STRING_VERSION(NDB_VERSION_MAJOR, \
                                 NDB_VERSION_MINOR, \
                                 NDB_VERSION_BUILD) NDB_VERSION_STATUS

/*
  The version number of the MySQL Server that NDB is built
  with. Extracted from MYSQL_VERSION_ID
*/
#define NDB_MYSQL_VERSION_MAJOR ((MYSQL_VERSION_ID / 10000) % 100)
#define NDB_MYSQL_VERSION_MINOR ((MYSQL_VERSION_ID /100) % 100)
#define NDB_MYSQL_VERSION_BUILD (MYSQL_VERSION_ID % 100)

/* Composite version number for MYSQL Server */
#define NDB_MYSQL_VERSION_D \
  NDB_MAKE_VERSION(NDB_MYSQL_VERSION_MAJOR, \
                   NDB_MYSQL_VERSION_MINOR, \
                   NDB_MYSQL_VERSION_BUILD)

#define NDB_VERSION_STRING_BUF_SZ 100

#ifdef __cplusplus
extern "C" {
#endif

void ndbPrintVersion();

Uint32 ndbMakeVersion(Uint32 major, Uint32 minor, Uint32 build);

Uint32 ndbGetMajor(Uint32 version);
  
Uint32 ndbGetMinor(Uint32 version);
  
Uint32 ndbGetBuild(Uint32 version);

const char* ndbGetVersionString(Uint32 version, Uint32 mysql_version, const char * status,
                                char *buf, unsigned sz);
const char* ndbGetOwnVersionString();

Uint32 ndbGetOwnVersion();

#ifdef __cplusplus
}
#endif

#define NDB_VERSION_STRING ndbGetOwnVersionString()

#define NDB_VERSION ndbGetOwnVersion()

/**
 * This is updated each time a version of backup/lcp format is changed
 *   when it's updated, it's set to version that made the change
 */
#define NDB_BACKUP_VERSION NDB_MAKE_VERSION(6,3,11)

/**
 * From which version do we support rowid
 */ 
#define NDBD_ROWID_VERSION (NDB_MAKE_VERSION(5,1,6))
#define NDBD_INCL_NODECONF_VERSION_4 NDB_MAKE_VERSION(4,1,17)
#define NDBD_INCL_NODECONF_VERSION_5 NDB_MAKE_VERSION(5,0,18)
#define NDBD_FRAGID_VERSION (NDB_MAKE_VERSION(5,1,6))
#define NDBD_DICT_LOCK_VERSION_5 NDB_MAKE_VERSION(5,0,23)
#define NDBD_DICT_LOCK_VERSION_5_1 NDB_MAKE_VERSION(5,1,12)

#define NDBD_UPDATE_FRAG_DIST_KEY_50 NDB_MAKE_VERSION(5,0,26)
#define NDBD_UPDATE_FRAG_DIST_KEY_51 NDB_MAKE_VERSION(5,1,12)

#define NDBD_QMGR_SINGLEUSER_VERSION_5 NDB_MAKE_VERSION(5,0,25)

#define NDBD_SPLIT_VERSION NDB_MAKE_VERSION(6,1,0)
#define NDBD_NODE_VERSION_REP NDB_MAKE_VERSION(6,1,1)

#define NDBD_255_NODES_VERSION NDB_MAKE_VERSION(5,1,4)

#define NDBD_MICRO_GCP_62 NDB_MAKE_VERSION(6,2,5)
#define NDBD_MICRO_GCP_63 NDB_MAKE_VERSION(6,3,2)
#define NDBD_RAW_LCP MAKE_VERSION(6,3,11)
#define NDBD_LONG_TCKEYREQ NDB_MAKE_VERSION(6,4,0)
#define NDBD_LONG_LQHKEYREQ MAKE_VERSION(6,4,0)
#define NDBD_MAX_RECVBYTESIZE_32K MAKE_VERSION(6,3,18)
#define NDBD_LONG_SCANTABREQ NDB_MAKE_VERSION(6,4,0)
#define NDBD_LONG_SCANFRAGREQ MAKE_VERSION(6,4,0)
#define NDBD_MT_LQH_VERSION MAKE_VERSION(6,4,0)

#define NDBD_SCHEMA_TRANS_VERSION NDB_MAKE_VERSION(6,4,0)

static
inline
int
ndb_check_micro_gcp(Uint32 version)
{	
  if (version == NDB_VERSION_D)
    return 1;
  {
    const Uint32 major = (version >> 16) & 0xFF;
    const Uint32 minor = (version >>  8) & 0xFF;
    if (major >= 6)
    {
      if (minor == 2)
        return version >= NDBD_MICRO_GCP_62;
      return version >= NDBD_MICRO_GCP_63;
    }
  }
  return 0;
}

#define NDBD_PREPARE_COPY_FRAG_VERSION NDB_MAKE_VERSION(6,2,1)
#define NDBD_PREPARE_COPY_FRAG_V2_51 NDB_MAKE_VERSION(5,1,23)
#define NDBD_PREPARE_COPY_FRAG_V2_62 NDB_MAKE_VERSION(6,2,8)
#define NDBD_PREPARE_COPY_FRAG_V2_63 NDB_MAKE_VERSION(6,3,6)

/**
 * 0 = NO PREP COPY FRAG SUPPORT
 * 1 = NO MAX PAGE SUPPORT
 * 2 = LATEST VERSION
 */
static
inline
int
ndb_check_prep_copy_frag_version(Uint32 version)
{	
  const Uint32 major = (version >> 16) & 0xFF;
  const Uint32 minor = (version >>  8) & 0xFF;

  if (version == NDB_VERSION_D)
    return 2;

  if (major >= 6)
  {
    if (minor == 2)
    {
      if (version >= NDBD_PREPARE_COPY_FRAG_V2_62)
        return 2;
      if (version >= NDBD_PREPARE_COPY_FRAG_VERSION)
        return 1;
      return 0;
    }
    else if (minor == 3)
    {
      if (version >= NDBD_PREPARE_COPY_FRAG_V2_63)
        return 2;
      return 1;
    }
    return 2;
  }
  else if (major == 5 && minor == 1)
  {
    if (version >= NDBD_PREPARE_COPY_FRAG_V2_51)
      return 2;
  }
    
  return 0;
}

#define NDBD_PNR NDB_MAKE_VERSION(6,3,8)

static 
inline
int
ndb_pnr(Uint32 version)
{
  return 
    version == NDB_VERSION_D || version >= NDBD_PNR;
}

#define NDBD_SUMA_DICT_LOCK_62 NDB_MAKE_VERSION(6,2,14)
#define NDBD_SUMA_DICT_LOCK_63 NDB_MAKE_VERSION(6,3,11)

static
inline
int
ndbd_suma_dictlock_startme(Uint32 x)
{
  if (x >= NDB_VERSION_D)
    return 1;
  
  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;
    
    if (major >= 6)
    {
      if (minor == 2)
      return x >= NDBD_SUMA_DICT_LOCK_62;
    }
    
    return x >= NDBD_SUMA_DICT_LOCK_63;
  }
}

#define NDBD_LONG_LIST_TABLES_CONF_62 NDB_MAKE_VERSION(6,2,16)
#define NDBD_LONG_LIST_TABLES_CONF_63 NDB_MAKE_VERSION(6,3,15)

static
inline
int
ndbd_LIST_TABLES_CONF_long_signal(Uint32 x)
{
  if (x >= NDB_VERSION_D)
    return 1;

  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;
    
    if (major >= 6)
    {
      if (minor == 2)
        return x >= NDBD_LONG_LIST_TABLES_CONF_62;
    }
    
    return x >= NDBD_LONG_LIST_TABLES_CONF_63;
  }
}

#if NOT_YET
#define NDBD_FIRE_TRIG_ORD_TRANSID_63 NDB_MAKE_VERSION(6,3,9)
#define NDBD_FIRE_TRIG_ORD_TRANSID_62 NDB_MAKE_VERSION(6,2,12)
#endif

static
inline
int
ndb_fire_trig_ord_transid(Uint32 version)
{
  if (version >= NDB_VERSION_D)
    return 1;

#if NOT_YET
  const Uint32 major = (version >> 16) & 0xFF;
  const Uint32 minor = (version >>  8) & 0xFF;
  if (major >= 6)
  {
    if (minor == 2)
    {
      return version >= NDBD_FIRE_TRIG_ORD_TRANSID_62;
    }

    return version >= NDBD_FIRE_TRIG_ORD_TRANSID_63;
  }
#endif

  return 0;
}

#define NDBD_SCAN_DISTKEY NDB_MAKE_VERSION(6,4,0)

static
inline
int
ndb_scan_distributionkey(Uint32 version)
{
  return version >= NDBD_SCAN_DISTKEY;
}

#define NDBD_FILTER_INSTANCE_63 NDB_MAKE_VERSION(6,3,16)
#define NDBD_COPY_GCI_RESTART_NR NDB_MAKE_VERSION(6,3,18)

#define NDBD_SUMA_DICTLOCK_HANDOVER NDB_MAKE_VERSION(6,4,0)

static
inline
int
ndbd_suma_dictlock_handover(Uint32 x)
{
  return (x >= NDBD_SUMA_DICTLOCK_HANDOVER);
}

#define NDBD_API_TAKE_OVERTCCONF_60 NDB_MAKE_VERSION(5,2,4)
#define NDBD_API_TAKE_OVERTCCONF_62 NDB_MAKE_VERSION(6,2,17)
#define NDBD_API_TAKE_OVERTCCONF_63 NDB_MAKE_VERSION(6,3,19)

#define NDBD_DELAYED_COPY_ACTIVEREQ_63 NDB_MAKE_VERSION(6,3,29)
#define NDBD_DELAYED_COPY_ACTIVEREQ_70 NDB_MAKE_VERSION(7,0,10)

static
inline
int
ndb_takeovertc(Uint32 x)
{
  if (x >= NDB_VERSION_D)
    return 1;

  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;
    
    if (major >= 6)
    {
      if (minor == 2)
        return x >= NDBD_API_TAKE_OVERTCCONF_62;
    }
    
    return x >= NDBD_API_TAKE_OVERTCCONF_63;
  }
}

static
inline
int
ndb_delayed_copy_active_req(Uint32 x)
{
  if (x >= NDB_VERSION_D)
    return 1;

  {
    const Uint32 major = (x >> 16) & 0xFF;

    if (major == 6)
    {
      return x >= NDBD_DELAYED_COPY_ACTIVEREQ_63;
    }
    else
    {
      return x >= NDBD_DELAYED_COPY_ACTIVEREQ_70;
    }
  }
}

#define NDBD_SUMA_NOTSTARTED_REF_63 NDB_MAKE_VERSION(6,3,29)
#define NDBD_SUMA_NOTSTARTED_REF_70 NDB_MAKE_VERSION(7,0,10)

static
inline
int
ndb_suma_not_started_ref(Uint32 x)
{
  if (x >= NDB_VERSION_D)
    return 1;

  {
    const Uint32 major = (x >> 16) & 0xFF;

    if (major == 6)
    {
      return x >= NDBD_SUMA_NOTSTARTED_REF_63;
    }
    else
    {
      return x >= NDBD_SUMA_NOTSTARTED_REF_70;
    }
  }
}

#define NDBD_UNLOCK_OP_SUPPORTED NDB_MAKE_VERSION(7,0,12)

#define NDBD_ROUTE_EXEC_FRAG NDB_MAKE_VERSION(7,0,13)

static
inline
int
ndb_route_exec_frag(Uint32 x)
{
  if (x >= NDB_VERSION_D)
    return 1;

  return x >= NDBD_ROUTE_EXEC_FRAG;
}

#define NDBD_NATIVE_DEFAULT_SUPPORT_70 NDB_MAKE_VERSION(7,0,15)
#define NDBD_NATIVE_DEFAULT_SUPPORT_71 NDB_MAKE_VERSION(7,1,4)

static
inline
int
ndb_native_default_support(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor == 0)
    return x >= NDBD_NATIVE_DEFAULT_SUPPORT_70;
  
  return x >= NDBD_NATIVE_DEFAULT_SUPPORT_71;
}

#define NDBD_HB_ORDER_VERSION_63 NDB_MAKE_VERSION(6,3,35)
#define NDBD_HB_ORDER_VERSION_70 NDB_MAKE_VERSION(7,0,16)
#define NDBD_HB_ORDER_VERSION_71 NDB_MAKE_VERSION(7,1,5)

static
inline
int
ndb_check_hb_order_version(Uint32 x)
{
  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;

    if (major == 6 && minor == 3)
    {
      return x >= NDBD_HB_ORDER_VERSION_63;
    }
    if (major == 7 && minor == 0)
    {
      return x >= NDBD_HB_ORDER_VERSION_70;
    }
    return x >= NDBD_HB_ORDER_VERSION_71;
  }
}

#define NDBD_SYNCH_SP_70 NDB_MAKE_VERSION(7,0,17)
#define NDBD_SYNCH_SP_71 NDB_MAKE_VERSION(7,1,6)

static
inline
int
ndb_wait_sp(Uint32 x)
{
  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;

    if (major == 7 && minor == 0)
    {
      return x >= NDBD_SYNCH_SP_70;
    }
    return x >= NDBD_SYNCH_SP_71;
  }
}

#define NDBD_DIH_GET_TABINFO_70 NDB_MAKE_VERSION(7,0,17)
#define NDBD_DIH_GET_TABINFO_71 NDB_MAKE_VERSION(7,1,6)

static
inline
int
ndb_dih_get_tabinfo(Uint32 x)
{
  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;

    if (major == 7 && minor == 0)
    {
      return x >= NDBD_DIH_GET_TABINFO_70;
    }
    return x >= NDBD_DIH_GET_TABINFO_71;
  }
}

#define NDBD_SUMA_STOP_ME_63 NDB_MAKE_VERSION(6,3,37)
#define NDBD_SUMA_STOP_ME_70 NDB_MAKE_VERSION(7,0,18)
#define NDBD_SUMA_STOP_ME_71 NDB_MAKE_VERSION(7,1,7)

static
inline
int
ndbd_suma_stop_me(Uint32 x)
{
  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;

    if (major == 6)
    {
      return x >= NDBD_SUMA_STOP_ME_63;
    }
    if (major == 7 && minor == 0)
    {
      return x >= NDBD_SUMA_STOP_ME_70;
    }
    return x >= NDBD_SUMA_STOP_ME_71;
  }
}

#define NDBD_DIH_SUB_GCP_COMPLETE_ACK_63 NDB_MAKE_VERSION(6,3,39)
#define NDBD_DIH_SUB_GCP_COMPLETE_ACK_70 NDB_MAKE_VERSION(7,0,20)
#define NDBD_DIH_SUB_GCP_COMPLETE_ACK_71 NDB_MAKE_VERSION(7,1,9)

static
inline
int
ndbd_dih_sub_gcp_complete_ack(Uint32 x)
{
  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;

    if (major == 6)
    {
      return x >= NDBD_DIH_SUB_GCP_COMPLETE_ACK_63;
    }
    if (major == 7 && minor == 0)
    {
      return x >= NDBD_DIH_SUB_GCP_COMPLETE_ACK_70;
    }
    return x >= NDBD_DIH_SUB_GCP_COMPLETE_ACK_71;
  }
}

/**
 * After this version, TC/BACKUP/SUMA does not use ScanFragReq::setAttrLen
 *   when sending long SCAN_FRAGREQ
 */
#define NDBD_LONG_SCAN_FRAGREQ_70 NDB_MAKE_VERSION(7,0,20)
#define NDBD_LONG_SCAN_FRAGREQ_71 NDB_MAKE_VERSION(7,1,9)

#define NDBD_FAIL_REP_SOURCE_NODE_63 NDB_MAKE_VERSION(6,3,40)
#define NDBD_FAIL_REP_SOURCE_NODE_70 NDB_MAKE_VERSION(7,0,21)
#define NDBD_FAIL_REP_SOURCE_NODE_71 NDB_MAKE_VERSION(7,1,10)

static 
inline
int
ndbd_fail_rep_source_node(Uint32 x)
{
  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;

    if (major == 6)
    {
      return x >= NDBD_FAIL_REP_SOURCE_NODE_63;
    }
    if (major == 7 && minor == 0)
    {
      return x >= NDBD_FAIL_REP_SOURCE_NODE_70;
    }
    return x >= NDBD_FAIL_REP_SOURCE_NODE_71;
  }
}


#define NDBD_SYNC_REQ_SUPPORT_70 NDB_MAKE_VERSION(7,0,20)
#define NDBD_SYNC_REQ_SUPPORT_71 NDB_MAKE_VERSION(7,1,9)

static
inline
int
ndbd_sync_req_support(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor == 0)
    return x >= NDBD_SYNC_REQ_SUPPORT_70;

  return x >= NDBD_SYNC_REQ_SUPPORT_71;
}

/**
 * Does not support CopyFragReq::CFR_NON_TRANSACTIONAL
 */
#define NDBD_NON_TRANS_COPY_FRAG_REQ_70 NDB_MAKE_VERSION(7,0,22)
#define NDBD_NON_TRANS_COPY_FRAG_REQ_71 NDB_MAKE_VERSION(7,1,11)

static
inline
int
ndbd_non_trans_copy_frag_req(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor == 0)
    return x >= NDBD_NON_TRANS_COPY_FRAG_REQ_70;

  return x >= NDBD_NON_TRANS_COPY_FRAG_REQ_71;
}

#define NDBD_PING_REQ_70 NDB_MAKE_VERSION(7,0,24)
#define NDBD_PING_REQ_71 NDB_MAKE_VERSION(7,1,13)

static
inline
int
ndbd_connectivity_check(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor == 0)
    return x >= NDBD_PING_REQ_70;

  return x >= NDBD_PING_REQ_71;
}

#define NDBD_DEFERRED_UNIQUE_CONSTRAINTS_70 NDB_MAKE_VERSION(7,0,25)
#define NDBD_DEFERRED_UNIQUE_CONSTRAINTS_71 NDB_MAKE_VERSION(7,1,14)

static
inline
int
ndbd_deferred_unique_constraints(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor == 0)
    return x >= NDBD_DEFERRED_UNIQUE_CONSTRAINTS_70;

  return x >= NDBD_DEFERRED_UNIQUE_CONSTRAINTS_71;
}

#define NDBD_TUP_EXTRABITS_70 NDB_MAKE_VERSION(7,0,25)
#define NDBD_TUP_EXTRABITS_71 NDB_MAKE_VERSION(7,1,14)
#define NDBD_TUP_EXTRABITS_72 NDB_MAKE_VERSION(7,2,1)

static
inline
int
ndb_tup_extrabits(Uint32 x)
{
  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;

    if (major == 7 && minor < 2)
    {
      if (minor == 0)
        return x >= NDBD_TUP_EXTRABITS_70;
      else if (minor == 1)
        return x >= NDBD_TUP_EXTRABITS_71;
    }
    return x >= NDBD_TUP_EXTRABITS_72;
  }
}

#define NDBD_REFRESH_TUPLE_70 NDB_MAKE_VERSION(7,0,26)
#define NDBD_REFRESH_TUPLE_71 NDB_MAKE_VERSION(7,1,15)
#define NDBD_REFRESH_TUPLE_72 NDB_MAKE_VERSION(7,2,1)

static
inline
int
ndb_refresh_tuple(Uint32 x)
{
  {
    const Uint32 major = (x >> 16) & 0xFF;
    const Uint32 minor = (x >>  8) & 0xFF;

    if (major == 7 && minor < 2)
    {
      if (minor == 0)
        return x >= NDBD_REFRESH_TUPLE_70;
      else if (minor == 1)
        return x >= NDBD_REFRESH_TUPLE_71;
    }
    return x >= NDBD_REFRESH_TUPLE_72;
  }
}

#define NDBD_GET_CONFIG_SUPPORT_70 NDB_MAKE_VERSION(7,0,27)
#define NDBD_GET_CONFIG_SUPPORT_71 NDB_MAKE_VERSION(7,1,16)

static
inline
int
ndbd_get_config_supported(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor == 0)
    return x >= NDBD_GET_CONFIG_SUPPORT_70;

  return x >= NDBD_GET_CONFIG_SUPPORT_71;
}

#define NDBD_CONFIGURABLE_LOG_PARTS_70 NDB_MAKE_VERSION(7,0,29)
#define NDBD_CONFIGURABLE_LOG_PARTS_71 NDB_MAKE_VERSION(7,1,18)
#define NDBD_CONFIGURABLE_LOG_PARTS_72 NDB_MAKE_VERSION(7,2,3)

static
inline
int
ndb_configurable_log_parts(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor < 2)
  {
    if (minor == 0)
      return x >= NDBD_CONFIGURABLE_LOG_PARTS_70;
    else if (minor == 1)
      return x >= NDBD_CONFIGURABLE_LOG_PARTS_71;
  }
  return x >= NDBD_CONFIGURABLE_LOG_PARTS_72;
}

#define NDBD_128_INSTANCES_ADDRESS_70 NDB_MAKE_VERSION(7,0,31)
#define NDBD_128_INSTANCES_ADDRESS_71 NDB_MAKE_VERSION(7,1,20)
#define NDBD_128_INSTANCES_ADDRESS_72 NDB_MAKE_VERSION(7,2,5)

static
inline
int
ndbd_128_instances_address(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor < 2)
  {
    if (minor == 0)
      return x >= NDBD_128_INSTANCES_ADDRESS_70;
    else if (minor == 1)
      return x >= NDBD_128_INSTANCES_ADDRESS_71;
  }
  return x >= NDBD_128_INSTANCES_ADDRESS_72;
}

#define NDBD_SCAN_TABREQ_IMPLICIT_PARALLELISM_70 NDB_MAKE_VERSION(7,0,34)
#define NDBD_SCAN_TABREQ_IMPLICIT_PARALLELISM_71 NDB_MAKE_VERSION(7,1,23)
#define NDBD_SCAN_TABREQ_IMPLICIT_PARALLELISM_72 NDB_MAKE_VERSION(7,2,7)

static
inline
int
ndbd_scan_tabreq_implicit_parallelism(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor < 2)
  {
    if (minor == 0)
      return x >= NDBD_SCAN_TABREQ_IMPLICIT_PARALLELISM_70;
    else if (minor == 1)
      return x >= NDBD_SCAN_TABREQ_IMPLICIT_PARALLELISM_71;
  }
  return x >= NDBD_SCAN_TABREQ_IMPLICIT_PARALLELISM_72;
}

#define NDBD_FIXED_LOOKUP_QUERY_ABORT_72 NDB_MAKE_VERSION(7,2,5)

static
inline
int
ndbd_fixed_lookup_query_abort(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major == 7 && minor < 2)
  {
    // Only experimental support of SPJ pre 7.2.0.
    // Assume we support 'fixed-abort' as we want it tested.
    return 1;
  }
  return x >= NDBD_FIXED_LOOKUP_QUERY_ABORT_72;
}

/**
 * NOTE1:
 *   Even though pushed join support wasn't GA intil 7.2.4
 *   we claim support for it in all 7.2.x versions.
 * NOTE2:
 *   By a mistake this online upgrade check was not
 *   added until version 7.2.6
 */
#define NDBD_JOIN_PUSHDOWN NDB_MAKE_VERSION(7,2,0)

static
inline
int
ndb_join_pushdown(Uint32 x)
{
  return x >= NDBD_JOIN_PUSHDOWN;
}

/**
 * DICT did not have a function for receiving GET_TABINFOREF signals
 * (see Bug#14647210).
 */
#define NDBD_DICT_GET_TABINFOREF_IMPLEMENTED NDB_MAKE_VERSION(7, 2, 9)

inline
int
ndbd_dict_get_tabinforef_implemented(Uint32 x)
{
  return x >= NDBD_DICT_GET_TABINFOREF_IMPLEMENTED;
}

#define NDBD_MULTI_TC_INSTANCE_TAKEOVER_72 NDB_MAKE_VERSION(7,2,17)
#define NDBD_MULTI_TC_INSTANCE_TAKEOVER_73 NDB_MAKE_VERSION(7,3,6)

static
inline
int
ndbd_multi_tc_instance_takeover(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major < 7)
  {
    return 0;
  }
  else if (major == 7)
  {
    if (minor < 2)
    {
      return 0;
    }
    else if (minor == 2)
    {
      return x >= NDBD_MULTI_TC_INSTANCE_TAKEOVER_72;
    }
    else if (minor == 3)
    {
      return x >= NDBD_MULTI_TC_INSTANCE_TAKEOVER_73;
    }
  }
  return 1;
}

#define NDBD_ISOLATE_ORD_72 NDB_MAKE_VERSION(7,2,19)
#define NDBD_ISOLATE_ORD_73 NDB_MAKE_VERSION(7,3,8)
#define NDBD_ISOLATE_ORD_74 NDB_MAKE_VERSION(7,4,3)


static
inline
int
ndbd_isolate_ord(Uint32 x)
{
  const Uint32 major = (x >> 16) & 0xFF;
  const Uint32 minor = (x >>  8) & 0xFF;

  if (major < 7)
  {
    return 0;
  }
  else if (major == 7)
  {
    if (minor < 2)
    {
      return 0;
    }
    else if (minor == 2)
    {
      return x >= NDBD_ISOLATE_ORD_72;
    }
    else if (minor == 3)
    {
      return x >= NDBD_ISOLATE_ORD_73;
    }
    else if (minor == 4)
    {
      return x >= NDBD_ISOLATE_ORD_74;
    }
  }
  return 1;
}


#endif
